
/*****************************************************************************/
/*                              Legal                                        */
/*****************************************************************************/

/*
** Copyright 2015-2025, Lantronix, Inc. All Rights Reserved.
** By using this software, you are agreeing to the terms of the Software
** Development Kit (SDK) License Agreement included in the distribution package
** for this software (the License Agreement).
** Under the License Agreement, this software may be used solely to create
** custom applications for use on the Lantronix xPico Wi-Fi product.
** THIS SOFTWARE AND ANY ACCOMPANYING DOCUMENTATION IS PROVIDED "AS IS".
** LANTRONIX SPECIFICALLY DISCLAIMS ANY WARRANTIES, INCLUDING, BUT NOT LIMITED
** TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY, NON-INFRINGEMENT AND FITNESS
** FOR A PARTICULAR PURPOSE.
** LANTRONIX HAS NO OBLIGATION TO PROVIDE MAINTENANCE, SUPPORT, UPDATES,
** ENHANCEMENTS, OR MODIFICATIONS TO THIS SOFTWARE.
** IN NO EVENT SHALL LANTRONIX BE LIABLE TO ANY PARTY FOR DIRECT, INDIRECT,
** SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES, INCLUDING LOST PROFITS,
** ARISING OUT OF THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF
** LANTRONIX HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
*/

/*****************************************************************************/
/*                           Documentation                                   */
/*****************************************************************************/

/*!
** \addtogroup example
** @{
*/

/*!
** \defgroup v_line v_line
** @{
**
** The \b v_line module implements a rudimentary "Line Protocol".
** When this protocol is chosen by a Line, it hosts Line Virtual_1.
** Data is transferred transparently in both directions.
** Modify this code to re-format data to or from the Line.
**
** Build it from project "vLineDemo".
*/

/*****************************************************************************/
/*                             Includes                                      */
/*****************************************************************************/

#include "v_line_module_defs.h" /* Automatically generated by make. */
#include "ltrx_line.h" /* Delivered with SDK. */
#include "ltrx_network.h" /* Delivered with SDK. */
#include "ltrx_stream.h" /* Delivered with SDK. */
#include "ltrx_tlog.h" /* Delivered with SDK. */
#include "virtual_line_module_libs.h" /* Delivered with SDK. */

/*****************************************************************************/
/*                             Structs                                       */
/*****************************************************************************/

struct thread_info
{
    uint32_t zeroBasedIndex;
    bool isRunning;
    struct ltrx_trigger eventTrigger;
    const struct ltrx_network_protocol *protocol;
    struct ltrx_network_protocol_handle *handle;
    enum ltrx_network_return netSendState;
};

/*****************************************************************************/
/*                              Globals                                      */
/*****************************************************************************/

const struct virtual_line_external_functions *g_virtual_lineExternalFunctionEntry_pointer = 0;

/*****************************************************************************/
/*                            Prototypes                                     */
/*****************************************************************************/

bool StartLineProtocol(uint16_t zeroBasedIndex);

void StopLineProtocol(uint16_t zeroBasedIndex);

/*****************************************************************************/
/*                         Local Constants                                   */
/*****************************************************************************/

static const struct ltrx_line_protocol s_lineProtocol =
{
    .protocolName = "V Line",
    .helpHtml = "SDK example.",
    .startProtocol = StartLineProtocol,
    .stopProtocol = StopLineProtocol
};

/*****************************************************************************/
/*                         Local Variables                                   */
/*****************************************************************************/

static struct thread_info *s_threadInfo[MAX_LOGICAL_SERIAL_LINES];

static struct ltrx_thread *s_threadForLine[MAX_LOGICAL_SERIAL_LINES];

/*****************************************************************************/
/*                               Code                                        */
/*****************************************************************************/

void v_line_module_registration(void)
{
    ltrx_module_register(&g_v_lineModuleInfo);
    ltrx_line_register_protocol(&s_lineProtocol);
}

void v_line_module_startup(void)
{
    g_virtual_lineExternalFunctionEntry_pointer = ltrx_module_functions_lookup("Virtual_Line");
}

void v_line_module_shutdown(void)
{
}

static void lineLoop(struct thread_info *ti)
{
    static const char s_openingMessage[] = "\r\nV Line demo\r\n";
    ltrx_line_write(ti->zeroBasedIndex, s_openingMessage, sizeof(s_openingMessage) - 1, &ti->eventTrigger); /* Non-blocking write */
    while(ti->isRunning)
    {
        ltrx_trigger_clear(&ti->eventTrigger);
        /*
        ** The line write with zero length returns true if the Line
        ** write has completed.
        ** If not, the event trigger will be signalled when completed.
        */
        if(ltrx_line_write(ti->zeroBasedIndex, NULL, 0, &ti->eventTrigger))
        {
            ti->protocol->protocolReceiveData(ti->handle, NULL, 0); /* Frees previous rx buffer. */
            const uint8_t *p;
            /* Use zero-copy APIs to move data from the network to line. */
            size_t n = ti->protocol->protocolReceiveData(ti->handle, &p, 0xffff); /* Gets rx buffer, if data is available. */
            if(n)
            {
                ltrx_line_write(ti->zeroBasedIndex, p, n, &ti->eventTrigger);
            }
        }
        if(ti->netSendState == LTRX_NETWORK_RETURN__WORKING)
        {
            ti->netSendState = ti->protocol->optProtocolSendDataNonBlocking(ti->handle, NULL, 0, false, &ti->eventTrigger);
        }
        if(ti->netSendState == LTRX_NETWORK_RETURN__SUCCESS)
        {
            ltrx_line_read(ti->zeroBasedIndex, NULL, 0, 0); /* Frees previous rx buffer. */
            uint16_t avail = ltrx_line_read_bytes_available(ti->zeroBasedIndex, &ti->eventTrigger);
            if(avail)
            {
                uint8_t *r = NULL;
                size_t canSend = ti->protocol->protocolCanSend(ti->handle);
                /* Use zero-copy APIs to move data from the line to network. */
                uint16_t nRx = ltrx_line_read(ti->zeroBasedIndex, &r, canSend, 0); /* Gets rx buffer. */
                if(nRx)
                {
                    ti->netSendState = ti->protocol->optProtocolSendDataNonBlocking(ti->handle, r, nRx, true, &ti->eventTrigger);
                }
            }
        }
        if(ti->netSendState == LTRX_NETWORK_RETURN__FAILURE)
        {
            TLOG(TLOG_SEVERITY_LEVEL__ERROR, "V Line send failure");
            ti->isRunning = false;
            break;
        }
        LTRX_TRIGGER_WAIT(&ti->eventTrigger, TIME_WAIT_FOREVER);
    }
}

static void lineThread(void *opaque)
{
    uint16_t zeroBasedIndex = (uint32_t)opaque;
    bool loggedStartMessage = false;
    struct thread_info ti =
    {
        .zeroBasedIndex = zeroBasedIndex,
        .isRunning = true
    };
    if(! ltrx_trigger_create(&ti.eventTrigger, "LineThread"))
    {
        return;
    }
    s_threadInfo[zeroBasedIndex] = &ti;
    ti.protocol = ltrx_virtual_line_protocol();
    if(! ti.protocol)
    {
        TLOG(TLOG_SEVERITY_LEVEL__CRITICAL, "Virtual lines not supported");
        ti.isRunning = false;
    }
    while(
        ti.isRunning &&
        ! ltrx_line_open(zeroBasedIndex, 1000)
    );
    if(ti.isRunning)
    {
        struct ltrx_virtual_line_constructor_info lvlci =
        {
            .zeroBasedVirtualLine = 0,
            .eventTrigger = &ti.eventTrigger
        };
        ti.handle = ti.protocol->protocolConstructor(
            ti.protocol, (void *)&lvlci
        );
        if(! ti.handle)
        {
            TLOG(TLOG_SEVERITY_LEVEL__CRITICAL, "Virtual line 1 already in use");
            ti.isRunning = false;
        }
    }
    if(ti.isRunning)
    {
        TLOG(
            TLOG_SEVERITY_LEVEL__INFORMATIONAL,
            "%s started on Line:%s",
            s_lineProtocol.protocolName,
            ltrx_line_get_registered_name(zeroBasedIndex)
        );
        loggedStartMessage = true;
        ltrx_line_set_dtr(zeroBasedIndex, true);
        lineLoop(&ti);
    }
    if(loggedStartMessage)
    {
        TLOG(
            TLOG_SEVERITY_LEVEL__INFORMATIONAL,
            "%s stopped on Line:%s",
            s_lineProtocol.protocolName,
            ltrx_line_get_registered_name(zeroBasedIndex)
        );
    }
    if(ti.protocol && ti.handle)
    {
        ti.protocol->protocolDestructor(ti.handle);
    }
    ltrx_line_close(zeroBasedIndex);
    s_threadInfo[zeroBasedIndex] = NULL;
    s_threadForLine[zeroBasedIndex] = 0;
    ltrx_trigger_destroy(&ti.eventTrigger);
}


bool StartLineProtocol(uint16_t zeroBasedIndex)
{
    if(s_threadInfo[zeroBasedIndex] || s_threadForLine[zeroBasedIndex])
    {
        TLOG(
            TLOG_SEVERITY_LEVEL__ERROR,
            "%s thread already running for Line:%s",
            s_lineProtocol.protocolName,
            ltrx_line_get_registered_name(zeroBasedIndex)
        );
        return false;
    }
    s_threadForLine[zeroBasedIndex] = ltrx_thread_create(
        s_lineProtocol.protocolName,
        lineThread,
        (void *)(uint32_t)zeroBasedIndex,
        5000
    );
    if(! s_threadForLine[zeroBasedIndex])
    {
        TLOG(
            TLOG_SEVERITY_LEVEL__ERROR,
            "Failed to create %s thread for Line:%s",
            s_lineProtocol.protocolName,
            ltrx_line_get_registered_name(zeroBasedIndex)
        );
        return false;
    }
    return true;
}

void StopLineProtocol(uint16_t zeroBasedIndex)
{
    bool wasRunning = false;
    if(zeroBasedIndex < MAX_LOGICAL_SERIAL_LINES)
    {
        ltrx_preemption_block();
        struct thread_info *ti = s_threadInfo[zeroBasedIndex];
        if(ti && ti->isRunning)
        {
            wasRunning = true;
            ti->isRunning = false;
        }
        ltrx_preemption_unblock();
        if(wasRunning)
        {
            struct ltrx_thread *lt;
            uint32_t tm = ltrx_timemark();
            while(
                (lt = s_threadForLine[zeroBasedIndex]) != NULL &&
                lt != ltrx_thread_id() &&
                ltrx_elapsed_time_current_ms(tm) < 2000
            )
            {
                if(ltrx_elapsed_time_current_ms(tm) >= 500)
                {
                    ltrx_line_purge(zeroBasedIndex);
                }
                ltrx_thread_sleep(100);
            }
        }
    }
}

/*!
** @}
*/

/*!
** @}
*/
