
/*****************************************************************************/
/*                              Legal                                        */
/*****************************************************************************/

/*
** Copyright 2015-2025, Lantronix, Inc. All Rights Reserved.
** By using this software, you are agreeing to the terms of the Software
** Development Kit (SDK) License Agreement included in the distribution package
** for this software (the License Agreement).
** Under the License Agreement, this software may be used solely to create
** custom applications for use on the Lantronix xPico Wi-Fi product.
** THIS SOFTWARE AND ANY ACCOMPANYING DOCUMENTATION IS PROVIDED "AS IS".
** LANTRONIX SPECIFICALLY DISCLAIMS ANY WARRANTIES, INCLUDING, BUT NOT LIMITED
** TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY, NON-INFRINGEMENT AND FITNESS
** FOR A PARTICULAR PURPOSE.
** LANTRONIX HAS NO OBLIGATION TO PROVIDE MAINTENANCE, SUPPORT, UPDATES,
** ENHANCEMENTS, OR MODIFICATIONS TO THIS SOFTWARE.
** IN NO EVENT SHALL LANTRONIX BE LIABLE TO ANY PARTY FOR DIRECT, INDIRECT,
** SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES, INCLUDING LOST PROFITS,
** ARISING OUT OF THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF
** LANTRONIX HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
*/

/*****************************************************************************/
/*                           Documentation                                   */
/*****************************************************************************/

/*!
** \addtogroup example
** @{
*/

/*!
** \defgroup udp_tunnel udp_tunnel
** @{
**
** The \b udp_tunnel module implements a "Line Protocol".
** When this protocol is chosen by a Line, it sends data received via UDP
** out on the Line. It sends data received on the Line out via UDP when it
** receives \<Enter\> or fills its 80 character buffer.
**
** As delivered in the \b SDK, the module listens for UDP packets on interface
** \b wlan0 addressed to the \b LOCAL_PORT 10001.
** When the first packet is received, it sends "Connected" to the Line.
** Subsequently it listens to the line for data to send out via UDP.
** The \#define \b REMOTE_ADDRESS may be
** "ap0", "eth0", or "wlan0" to operate in this listening mode.
**
** Alternately, you can change the \#define \b REMOTE_ADDRESS to either an
** IP address or a host name to operate in a connect mode.
** In this case you must also set the \#define \b REMOTE_PORT.
** When the interface is established and resolves the name or address, it
** sends "Connected" to the Line.
**
** Build it from project "udpTunnelDemo".
*/

/*****************************************************************************/
/*                             Includes                                      */
/*****************************************************************************/

#include "ltrx_line.h" /* Delivered with SDK. */
#include "ltrx_network.h" /* Delivered with SDK. */
#include "ltrx_stream.h" /* Delivered with SDK. */
#include "ltrx_tlog.h" /* Delivered with SDK. */
#include "udp_tunnel_module_defs.h" /* Automatically generated by make. */

/*****************************************************************************/
/*                              Defines                                      */
/*****************************************************************************/

#define LOCAL_PORT 10001

/*
** For connecting out, fill in REMOTE_ADDRESS with either an IP address or
** a host name.
** For listening, fill in REMOTE_ADDRESS with either ap0 or wlan0, selecting
** the interface you want to listen on.
*/
#define REMOTE_ADDRESS "wlan0"

/*
** The REMOTE_PORT is only used if "connecting out" according to
** REMOTE_ADDRESS.
** If "listening", the remote port is derived from the most recent received
** packet "from" port.
*/
#define REMOTE_PORT 10001

/*****************************************************************************/
/*                             Structs                                       */
/*****************************************************************************/

struct thread_info
{
    uint32_t zeroBasedIndex;
    uint32_t loopTimeout;
    bool isRunning;
    bool isConnected;
    struct ltrx_trigger eventTrigger;
    struct output_stream_to_uart ostu;
    struct input_stream_from_uart isfu;
    struct ltrx_ip_socket *socket;
    size_t reportLength;
    uint8_t report[80];
};

/*****************************************************************************/
/*                            Prototypes                                     */
/*****************************************************************************/

bool StartLineProtocol(uint16_t zeroBasedIndex);

void StopLineProtocol(uint16_t zeroBasedIndex);

/*****************************************************************************/
/*                         Local Constants                                   */
/*****************************************************************************/

static const struct ltrx_line_protocol s_lineProtocol =
{
    .protocolName = "UDP Tunnel",
    .helpHtml = "SDK example.",
    .startProtocol = StartLineProtocol,
    .stopProtocol = StopLineProtocol
};

/*****************************************************************************/
/*                         Local Variables                                   */
/*****************************************************************************/

static struct thread_info *s_threadInfo[MAX_LOGICAL_SERIAL_LINES];

static struct ltrx_thread *s_threadForLine[MAX_LOGICAL_SERIAL_LINES];

/*****************************************************************************/
/*                               Code                                        */
/*****************************************************************************/

void udp_tunnel_module_registration(void)
{
    ltrx_module_register(&g_udp_tunnelModuleInfo);
    ltrx_line_register_protocol(&s_lineProtocol);
}

void udp_tunnel_module_startup(void)
{
}

void udp_tunnel_module_shutdown(void)
{
}

static void networkReadAndSerialWrite(struct thread_info *ti)
{
    uint8_t *receivedData;
    size_t length = ltrx_ip_socket_receive(
        ti->socket, &receivedData, 0xFFFFFFFF
    );
    if(length > 0)
    {
        ltrx_output_stream_write_binary(
            &ti->ostu.outStream, receivedData, length
        );
    }
}

static void serialReadEchoAndNetworkWrite(struct thread_info *ti)
{
    bool sendNow = false;
    while(ltrx_input_stream_peek(&ti->isfu.inStream) >= 0)
    {
        char c = ltrx_input_stream_read(&ti->isfu.inStream);
        if(c == '\r')
        {
            ltrx_output_stream_write_line(&ti->ostu.outStream, "");
            sendNow = true;
        }
        else
        {
            /* Echo. */
            ltrx_output_stream_write_binary(&ti->ostu.outStream, &c, 1);
            ti->report[ti->reportLength++] = c;
        }
        if(sendNow || ti->reportLength == sizeof(ti->report))
        {
            if(
                ti->reportLength &&
                ! ltrx_ip_socket_send(
                    ti->socket,
                    ti->report,
                    ti->reportLength,
                    true
                )
            )
            {
                ltrx_udp_socket_close(ti->socket);
                ti->socket = NULL;
                ti->loopTimeout = 5000;
                return;
            }
            sendNow = false;
            ti->reportLength = 0;
        }
    }
}

static void lineLoop(struct thread_info *ti)
{
    ltrx_input_stream_init_from_uart(&ti->isfu, ti->zeroBasedIndex);
    ltrx_output_stream_init_to_uart(&ti->ostu, ti->zeroBasedIndex);
    ltrx_output_stream_write_line(&ti->ostu.outStream, "");
    ltrx_output_stream_write_line(
        &ti->ostu.outStream, s_lineProtocol.protocolName
    );
    while(ti->isRunning)
    {
        /*
        ** We are using the event trigger on both serial receive and network
        ** receive, so by default there will be no loop timeout.
        ** However, in error cases we set the loop timeout for an appropriate
        ** retry time.
        */
        ti->loopTimeout = TIME_WAIT_FOREVER;
        struct ltrx_udp_socket_options options =
        {
            .optLocalPort = LOCAL_PORT,
            .optRemoteAddress = REMOTE_ADDRESS,
            .optRemotePort = REMOTE_PORT,
            .optReceiveEventTrigger = &ti->eventTrigger
        };
        if(
            ! ti->socket &&
            ! (
                ti->socket = ltrx_udp_socket_open_with_options(&options)
            )
        )
        {
            /* Failed to open UDP socket; try again later. */
            ti->loopTimeout = 5000;
        }
        else
        {
            networkReadAndSerialWrite(ti);
            if(ltrx_udp_socket_is_active(ti->socket))
            {
                if(! ti->isConnected)
                {
                    ltrx_output_stream_write_line(&ti->ostu.outStream, "");
                    ltrx_output_stream_write_line(&ti->ostu.outStream, "Connected");
                    ti->isConnected = true;
                }
                if(
                    ltrx_line_read_bytes_available(
                        ti->zeroBasedIndex, &ti->eventTrigger
                    )
                )
                {
                    serialReadEchoAndNetworkWrite(ti);
                }
            }
        }
        LTRX_TRIGGER_WAIT(&ti->eventTrigger, ti->loopTimeout);
        ltrx_trigger_clear(&ti->eventTrigger);
    }
}

static void lineThread(void *opaque)
{
    uint16_t zeroBasedIndex = (uint32_t)opaque;
    uint16_t line = zeroBasedIndex + 1;
    bool loggedStartMessage = false;
    struct thread_info ti =
    {
        .zeroBasedIndex = zeroBasedIndex,
        .isRunning = true
    };
    if(! ltrx_trigger_create(&ti.eventTrigger, s_lineProtocol.protocolName))
    {
        return;
    }
    s_threadInfo[zeroBasedIndex] = &ti;
    while(
        ti.isRunning &&
        ! ltrx_line_open(zeroBasedIndex, 1000)
    );
    if(ti.isRunning)
    {
        TLOG(
            TLOG_SEVERITY_LEVEL__INFORMATIONAL,
            "%s started on Line %u",
            s_lineProtocol.protocolName,
            line
        );
        loggedStartMessage = true;
        ltrx_line_set_dtr(zeroBasedIndex, true);
        lineLoop(&ti);
    }
    if(loggedStartMessage)
    {
        TLOG(
            TLOG_SEVERITY_LEVEL__INFORMATIONAL,
            "%s stopped on Line %u",
            s_lineProtocol.protocolName,
            line
        );
    }
    ltrx_line_close(zeroBasedIndex);
    s_threadInfo[zeroBasedIndex] = NULL;
    ltrx_trigger_destroy(&ti.eventTrigger);
    s_threadForLine[zeroBasedIndex] = 0;
}

bool StartLineProtocol(uint16_t zeroBasedIndex)
{
    uint16_t line = zeroBasedIndex + 1;
    if(s_threadInfo[zeroBasedIndex] || s_threadForLine[zeroBasedIndex])
    {
        TLOG(
            TLOG_SEVERITY_LEVEL__ERROR,
            "%s thread already running for Line %u",
            s_lineProtocol.protocolName,
            line
        );
        return false;
    }
    s_threadForLine[zeroBasedIndex] = ltrx_thread_create(
        s_lineProtocol.protocolName,
        lineThread,
        (void *)(uint32_t)zeroBasedIndex,
        5000
    );
    if(! s_threadForLine[zeroBasedIndex])
    {
        TLOG(
            TLOG_SEVERITY_LEVEL__ERROR,
            "Failed to create %s thread for Line %u",
            s_lineProtocol.protocolName,
            line
        );
        return false;
    }
    return true;
}

void StopLineProtocol(uint16_t zeroBasedIndex)
{
    bool wasRunning = false;
    if(zeroBasedIndex < MAX_LOGICAL_SERIAL_LINES)
    {
        ltrx_preemption_block();
        struct thread_info *ti = s_threadInfo[zeroBasedIndex];
        if(ti && ti->isRunning)
        {
            wasRunning = true;
            ti->isRunning = false;
            ltrx_trigger_signal(&ti->eventTrigger);
        }
        ltrx_preemption_unblock();
        if(wasRunning)
        {
            struct ltrx_thread *lt;
            uint32_t tm = ltrx_timemark();
            while(
                (lt = s_threadForLine[zeroBasedIndex]) != NULL &&
                lt != ltrx_thread_id() &&
                ltrx_elapsed_time_current_ms(tm) < 2000
            )
            {
                if(ltrx_elapsed_time_current_ms(tm) >= 500)
                {
                    ltrx_line_purge(zeroBasedIndex);
                }
                ltrx_thread_sleep(100);
            }
        }
    }
}

/*!
** @}
*/

/*!
** @}
*/
