
/*****************************************************************************/
/*                              Legal                                        */
/*****************************************************************************/

/*
** Copyright 2015-2025, Lantronix, Inc. All Rights Reserved.
** By using this software, you are agreeing to the terms of the Software
** Development Kit (SDK) License Agreement included in the distribution package
** for this software (the License Agreement).
** Under the License Agreement, this software may be used solely to create
** custom applications for use on the Lantronix xPico Wi-Fi product.
** THIS SOFTWARE AND ANY ACCOMPANYING DOCUMENTATION IS PROVIDED "AS IS".
** LANTRONIX SPECIFICALLY DISCLAIMS ANY WARRANTIES, INCLUDING, BUT NOT LIMITED
** TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY, NON-INFRINGEMENT AND FITNESS
** FOR A PARTICULAR PURPOSE.
** LANTRONIX HAS NO OBLIGATION TO PROVIDE MAINTENANCE, SUPPORT, UPDATES,
** ENHANCEMENTS, OR MODIFICATIONS TO THIS SOFTWARE.
** IN NO EVENT SHALL LANTRONIX BE LIABLE TO ANY PARTY FOR DIRECT, INDIRECT,
** SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES, INCLUDING LOST PROFITS,
** ARISING OUT OF THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF
** LANTRONIX HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
*/

/*****************************************************************************/
/*                           Documentation                                   */
/*****************************************************************************/

/*!
** \addtogroup example
** @{
*/

/*!
** \defgroup config_pin config_pin
** @{
**
** The \b config_pin module implements an output "Role" which can be assigned
** to any configurable pin.
**
** Build it from project "configurablePinDemo".
**
** Use either the Web Manager or Command Line Interface to setup configuration
** in group \b CPM.
** Under "Roles", find your role of "Blinker".
** Assign it a CP of your choice and set State to Enabled.
** (If some other role is enabled using your selected CP, this will fail.)
** Observe the blinking with an LED or scope connected to your selected CP.
**
** Notice use of the macro STACK_SIZE_GREEN_FROM_MAX_OBSERVED_STACK_USED
** when calling ltrx_thread_create().
** It was originally built with 2000 bytes of stack.
** Then the module was built and run.
** The actual stack was observed either via the Web Manager or the Command
** Line Interface under Diagnostics Threads.
** Find the "Blinker" thread, and the "Stack used" is shown.
** That value was plugged into the macro in the sample code.
*/

/*****************************************************************************/
/*                             Includes                                      */
/*****************************************************************************/

#include "config_pin_module_defs.h" /* Automatically generated by make. */
#include "ltrx_cpm.h" /* Delivered with SDK. */
#include "ltrx_tlog.h" /* Delivered with SDK. */

/*****************************************************************************/
/*                         Local Constants                                   */
/*****************************************************************************/

static const struct ltrx_cpm_role s_role =
{
	.name = "Blinker",
    .helpHtml = "SDK example",
	.usage = LTRX_CPM_ROLE_USAGE__OUTPUT
};

static const uint32_t s_delayTimeInMilliseconds = 500;

/*****************************************************************************/
/*                               Code                                        */
/*****************************************************************************/

static void blinkerThread(void *opaque)
{
    (void)opaque;
    uint16_t roleIndex = ltrx_cp_role_index_get(s_role.name);
    bool cpIsActive = false;
    if(roleIndex >= CPM_ROLE_MAX)
    {
        TLOG(
            TLOG_SEVERITY_LEVEL__ALERT,
            "%s role was not registered.", s_role.name
        );
        return;
    }
    TLOG(
        TLOG_SEVERITY_LEVEL__INFORMATIONAL,
        "%s role has logical index %u.", s_role.name, roleIndex
    );
    while(true)
    {
        ltrx_cp_write(roleIndex, cpIsActive);
        cpIsActive = ! cpIsActive;
        ltrx_thread_sleep(s_delayTimeInMilliseconds);
    }
}

void config_pin_module_registration(void)
{
    ltrx_module_register(&g_config_pinModuleInfo);
    ltrx_cp_register_role(&s_role);
}

void config_pin_module_startup(void)
{
    ltrx_thread_create(
        "Blinker",
        blinkerThread,
        NULL,
        STACK_SIZE_GREEN_FROM_MAX_OBSERVED_STACK_USED(247)
    );
}

void config_pin_module_shutdown(void)
{
}

/*!
** @}
*/

/*!
** @}
*/
