
/*****************************************************************************/
/*                              Legal                                        */
/*****************************************************************************/

/*
** Copyright 2015-2025, Lantronix, Inc. All Rights Reserved.
** By using this software, you are agreeing to the terms of the Software
** Development Kit (SDK) License Agreement included in the distribution package
** for this software (the License Agreement).
** Under the License Agreement, this software may be used solely to create
** custom applications for use on the Lantronix xPico Wi-Fi product.
** THIS SOFTWARE AND ANY ACCOMPANYING DOCUMENTATION IS PROVIDED "AS IS".
** LANTRONIX SPECIFICALLY DISCLAIMS ANY WARRANTIES, INCLUDING, BUT NOT LIMITED
** TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY, NON-INFRINGEMENT AND FITNESS
** FOR A PARTICULAR PURPOSE.
** LANTRONIX HAS NO OBLIGATION TO PROVIDE MAINTENANCE, SUPPORT, UPDATES,
** ENHANCEMENTS, OR MODIFICATIONS TO THIS SOFTWARE.
** IN NO EVENT SHALL LANTRONIX BE LIABLE TO ANY PARTY FOR DIRECT, INDIRECT,
** SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES, INCLUDING LOST PROFITS,
** ARISING OUT OF THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF
** LANTRONIX HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
*/

/*****************************************************************************/
/*                           Documentation                                   */
/*****************************************************************************/

/*!
** \addtogroup example
** @{
*/

/*!
** \defgroup spi_log spi_log
** @{
**
** The \b spi_log module implements implements a rudimentary "SPI Protocol".
** When this protocol is chosen by a SPI, it logs whatever it receives on the
** SPI.
**
** Build it from project "spiLogDemo".
**
** To run it you will need to select protocol "Log" under SPI Configuration.
**
** Also, you will need to enable the SPI CPs: Under CPM Roles Configuration,
** enable the SPI group of CPs.
** HINT: If using the evaluation board, you may place a jumper between JP5
** pins CP3 and CP4 to effect a loopback of your SPI data.
** Note that the SPI group of CPs is ganged, so when you enable one, you
** enable them all.
** Also note that all the other CPM Roles referring to any of the SPI CPs must
** be disabled.
*/

/*****************************************************************************/
/*                             Includes                                      */
/*****************************************************************************/

#include <stdio.h>

#include "ltrx_spi.h" /* Delivered with SDK. */
#include "ltrx_tlog.h" /* Delivered with SDK. */
#include "spi_log_module_defs.h" /* Automatically generated by make. */

/*****************************************************************************/
/*                              Defines                                      */
/*****************************************************************************/

#define MAX_TRANSFER_LENGTH 20

/*****************************************************************************/
/*                             Structs                                       */
/*****************************************************************************/

struct thread_info
{
    uint32_t zeroBasedIndex;
    bool isRunning;
};

/*****************************************************************************/
/*                            Prototypes                                     */
/*****************************************************************************/

bool StartSpiLog(uint16_t zeroBasedIndex);

void StopSpiLog(uint16_t zeroBasedIndex);

/*****************************************************************************/
/*                         Local Constants                                   */
/*****************************************************************************/

static const struct ltrx_spi_protocol s_spiProtocol = {
	.protocolName  = "Log",
    .helpHtml = "SDK example.",
	.startProtocol = StartSpiLog,
	.stopProtocol  = StopSpiLog
};

/*****************************************************************************/
/*                         Local Variables                                   */
/*****************************************************************************/

static struct thread_info *s_threadInfo[MAX_SPI_EXTERNAL];

static struct ltrx_thread *s_threadForSpi[MAX_SPI_EXTERNAL];

/*****************************************************************************/
/*                               Code                                        */
/*****************************************************************************/

void spi_log_module_registration(void)
{
    ltrx_module_register(&g_spi_logModuleInfo);
    ltrx_spi_register_protocol(&s_spiProtocol);
}

void spi_log_module_startup(void)
{
}

void spi_log_module_shutdown(void)
{
}

static void spiLoop(struct thread_info *ti)
{
    unsigned long count = 0;
    while(ti->isRunning)
    {
        char txBuffer[MAX_TRANSFER_LENGTH];
        uint8_t rxBuffer[MAX_TRANSFER_LENGTH];
        struct ltrx_spi_descriptor lsd =
        {
            .data_bytes = snprintf(txBuffer, sizeof(txBuffer), "Hello %08lx", ++count),
            .mosi_buf = (void *)txBuffer,
            .miso_buf = rxBuffer
        };
        ltrx_spi_transfer(ti->zeroBasedIndex, &lsd);
        TLOG(TLOG_SEVERITY_LEVEL__INFORMATIONAL, "Sent \"%s\"", txBuffer);
        TLOG_HEXDUMP(TLOG_SEVERITY_LEVEL__INFORMATIONAL, rxBuffer, lsd.data_bytes, NULL);
        ltrx_thread_sleep(5000);
    }
}

static void spiThread(void *opaque)
{
    uint16_t zeroBasedIndex = (uint32_t)opaque;
    uint16_t spi = zeroBasedIndex + 1;
    bool loggedStartMessage = false;
    struct thread_info ti =
    {
        .zeroBasedIndex = zeroBasedIndex,
        .isRunning = true
    };
    s_threadInfo[zeroBasedIndex] = &ti;
    while(
        ti.isRunning &&
        ! ltrx_spi_open(zeroBasedIndex, 1000)
    );
    if(ti.isRunning)
    {
        TLOG(
            TLOG_SEVERITY_LEVEL__INFORMATIONAL,
            "%s started on SPI %u",
            s_spiProtocol.protocolName,
            spi
        );
        loggedStartMessage = true;
    }
    if(ti.isRunning)
    {
        spiLoop(&ti);
    }
    if(loggedStartMessage)
    {
        TLOG(
            TLOG_SEVERITY_LEVEL__INFORMATIONAL,
            "%s stopped on SPI %u",
            s_spiProtocol.protocolName,
            spi
        );
    }
    ltrx_spi_close(zeroBasedIndex);
    s_threadInfo[zeroBasedIndex] = NULL;
    s_threadForSpi[zeroBasedIndex] = 0;
}

bool StartSpiLog(uint16_t zeroBasedIndex)
{
    uint16_t spi = zeroBasedIndex + 1;
    if(s_threadInfo[zeroBasedIndex] || s_threadForSpi[zeroBasedIndex])
    {
        TLOG(
            TLOG_SEVERITY_LEVEL__ERROR,
            "%s thread already running for SPI %u",
            s_spiProtocol.protocolName,
            spi
        );
        return false;
    }
    s_threadForSpi[zeroBasedIndex] = ltrx_thread_create(
        s_spiProtocol.protocolName,
        spiThread,
        (void *)(uint32_t)zeroBasedIndex,
        3000
    );
    if(! s_threadForSpi[zeroBasedIndex])
    {
        TLOG(
            TLOG_SEVERITY_LEVEL__ERROR,
            "Failed to create %s thread for SPI %u",
            s_spiProtocol.protocolName,
            spi
        );
        return false;
    }
    return true;
}

void StopSpiLog(uint16_t zeroBasedIndex)
{
    bool wasRunning = false;
    struct thread_info *ti;
    ltrx_preemption_block();
    ti = s_threadInfo[zeroBasedIndex];
    if(ti && ti->isRunning)
    {
        wasRunning = true;
        ti->isRunning = false;
    }
    ltrx_preemption_unblock();
    if(wasRunning)
    {
        struct ltrx_thread *lt;
        uint32_t tm = ltrx_timemark();
        while(
            (lt = s_threadForSpi[zeroBasedIndex]) != NULL &&
            lt != ltrx_thread_id() &&
            ltrx_elapsed_time_current_ms(tm) < 2000
        )
        {
            ltrx_thread_sleep(100);
        }
    }
}

/*!
** @}
*/

/*!
** @}
*/
