
/*****************************************************************************/
/*                              Legal                                        */
/*****************************************************************************/

/*
** Copyright 2015-2025, Lantronix, Inc. All Rights Reserved.
** By using this software, you are agreeing to the terms of the Software
** Development Kit (SDK) License Agreement included in the distribution package
** for this software (the License Agreement).
** Under the License Agreement, this software may be used solely to create
** custom applications for use on the Lantronix xPico Wi-Fi product.
** THIS SOFTWARE AND ANY ACCOMPANYING DOCUMENTATION IS PROVIDED "AS IS".
** LANTRONIX SPECIFICALLY DISCLAIMS ANY WARRANTIES, INCLUDING, BUT NOT LIMITED
** TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY, NON-INFRINGEMENT AND FITNESS
** FOR A PARTICULAR PURPOSE.
** LANTRONIX HAS NO OBLIGATION TO PROVIDE MAINTENANCE, SUPPORT, UPDATES,
** ENHANCEMENTS, OR MODIFICATIONS TO THIS SOFTWARE.
** IN NO EVENT SHALL LANTRONIX BE LIABLE TO ANY PARTY FOR DIRECT, INDIRECT,
** SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES, INCLUDING LOST PROFITS,
** ARISING OUT OF THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF
** LANTRONIX HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
*/

/*****************************************************************************/
/*                           Documentation                                   */
/*****************************************************************************/

/*!
** \addtogroup example
** @{
*/

/*!
** \defgroup line_echo line_echo
** @{
**
** The \b line_echo module implements a rudimentary "Line Protocol".
** When this protocol is chosen by a Line, it echoes back to the Line
** whatever it receives on the Line.
**
** Build it from project "echoDemo".
*/

/*****************************************************************************/
/*                             Includes                                      */
/*****************************************************************************/

#include "line_echo_module_defs.h" /* Automatically generated by make. */
#include "ltrx_line.h" /* Delivered with SDK. */
#include "ltrx_stream.h" /* Delivered with SDK. */
#include "ltrx_tlog.h" /* Delivered with SDK. */

/*****************************************************************************/
/*                             Structs                                       */
/*****************************************************************************/

struct thread_info
{
    uint32_t zeroBasedIndex;
    bool isRunning;
    struct ltrx_trigger eventTrigger;
    struct output_stream_to_uart ostu;
    struct input_stream_from_uart isfu;
};

/*****************************************************************************/
/*                            Prototypes                                     */
/*****************************************************************************/

bool StartLineProtocol(uint16_t zeroBasedIndex);

void StopLineProtocol(uint16_t zeroBasedIndex);

/*****************************************************************************/
/*                         Local Constants                                   */
/*****************************************************************************/

static const struct ltrx_line_protocol s_lineProtocol =
{
    .protocolName = "Echo",
    .helpHtml = "SDK example.",
    .startProtocol = StartLineProtocol,
    .stopProtocol = StopLineProtocol
};

/*****************************************************************************/
/*                         Local Variables                                   */
/*****************************************************************************/

static struct thread_info *s_threadInfo[MAX_LOGICAL_SERIAL_LINES];

static struct ltrx_thread *s_threadForLine[MAX_LOGICAL_SERIAL_LINES];

/*****************************************************************************/
/*                               Code                                        */
/*****************************************************************************/

void line_echo_module_registration(void)
{
    ltrx_module_register(&g_line_echoModuleInfo);
    ltrx_line_register_protocol(&s_lineProtocol);
}

void line_echo_module_startup(void)
{
}

void line_echo_module_shutdown(void)
{
}

static void echoData(struct thread_info *ti)
{
    while(ltrx_input_stream_peek(&ti->isfu.inStream) >= 0)
    {
        char c = ltrx_input_stream_read(&ti->isfu.inStream);
        {
            if(c == '\r')
            {
                ltrx_output_stream_write_line(&ti->ostu.outStream, "");
            }
            else
            {
                /* Echo. */
                ltrx_output_stream_write_binary(&ti->ostu.outStream, &c, 1);
            }
        }
    }
}

static void lineLoop(struct thread_info *ti)
{
    ltrx_input_stream_init_from_uart(&ti->isfu, ti->zeroBasedIndex);
    ltrx_output_stream_init_to_uart(&ti->ostu, ti->zeroBasedIndex);
    ltrx_output_stream_write_line(&ti->ostu.outStream, "");
    ltrx_output_stream_write_line(
        &ti->ostu.outStream, s_lineProtocol.protocolName
    );
    ltrx_output_stream_write_line(&ti->ostu.outStream, "Ready");
    while(ti->isRunning)
    {
        ltrx_trigger_clear(&ti->eventTrigger);
        if(
            ltrx_line_read_bytes_available(
                ti->zeroBasedIndex, &ti->eventTrigger
            )
        )
        {
            echoData(ti);
        }
        else
        {
            LTRX_TRIGGER_WAIT(&ti->eventTrigger, TIME_WAIT_FOREVER);
        }
    }
}

static void lineThread(void *opaque)
{
    uint16_t zeroBasedIndex = (uint32_t)opaque;
    uint16_t line = zeroBasedIndex + 1;
    bool loggedStartMessage = false;
    struct thread_info ti =
    {
        .zeroBasedIndex = zeroBasedIndex,
        .isRunning = true
    };
    if(! ltrx_trigger_create(&ti.eventTrigger, s_lineProtocol.protocolName))
    {
        return;
    }
    s_threadInfo[zeroBasedIndex] = &ti;
    while(
        ti.isRunning &&
        ! ltrx_line_open(zeroBasedIndex, 1000)
    );
    if(ti.isRunning)
    {
        TLOG(
            TLOG_SEVERITY_LEVEL__INFORMATIONAL,
            "%s started on Line %u",
            s_lineProtocol.protocolName,
            line
        );
        loggedStartMessage = true;
        ltrx_line_set_dtr(zeroBasedIndex, true);
        lineLoop(&ti);
    }
    if(loggedStartMessage)
    {
        TLOG(
            TLOG_SEVERITY_LEVEL__INFORMATIONAL,
            "%s stopped on Line %u",
            s_lineProtocol.protocolName,
            line
        );
    }
    ltrx_line_close(zeroBasedIndex);
    s_threadInfo[zeroBasedIndex] = NULL;
    ltrx_trigger_destroy(&ti.eventTrigger);
    s_threadForLine[zeroBasedIndex] = 0;
}

bool StartLineProtocol(uint16_t zeroBasedIndex)
{
    uint16_t line = zeroBasedIndex + 1;
    if(s_threadInfo[zeroBasedIndex] || s_threadForLine[zeroBasedIndex])
    {
        TLOG(
            TLOG_SEVERITY_LEVEL__ERROR,
            "%s thread already running for Line %u",
            s_lineProtocol.protocolName,
            line
        );
        return false;
    }
    s_threadForLine[zeroBasedIndex] = ltrx_thread_create(
        s_lineProtocol.protocolName,
        lineThread,
        (void *)(uint32_t)zeroBasedIndex,
        5000
    );
    if(! s_threadForLine[zeroBasedIndex])
    {
        TLOG(
            TLOG_SEVERITY_LEVEL__ERROR,
            "Failed to create %s thread for Line %u",
            s_lineProtocol.protocolName,
            line
        );
        return false;
    }
    return true;
}

void StopLineProtocol(uint16_t zeroBasedIndex)
{
    bool wasRunning = false;
    if(zeroBasedIndex < MAX_LOGICAL_SERIAL_LINES)
    {
        ltrx_preemption_block();
        struct thread_info *ti = s_threadInfo[zeroBasedIndex];
        if(ti && ti->isRunning)
        {
            wasRunning = true;
            ti->isRunning = false;
            ltrx_trigger_signal(&ti->eventTrigger);
        }
        ltrx_preemption_unblock();
        if(wasRunning)
        {
            struct ltrx_thread *lt;
            uint32_t tm = ltrx_timemark();
            while(
                (lt = s_threadForLine[zeroBasedIndex]) != NULL &&
                lt != ltrx_thread_id() &&
                ltrx_elapsed_time_current_ms(tm) < 2000
            )
            {
                if(ltrx_elapsed_time_current_ms(tm) >= 500)
                {
                    ltrx_line_purge(zeroBasedIndex);
                }
                ltrx_thread_sleep(100);
            }
        }
    }
}

/*!
** @}
*/

/*!
** @}
*/
