/*****************************************************************************/
/*                              Legal                                        */
/*****************************************************************************/

/*
** Copyright ©2015-2025, Lantronix, Inc. All Rights Reserved.
** By using this software, you are agreeing to the terms of the Software
** Development Kit (SDK) License Agreement included in the distribution package
** for this software (the “License Agreement”).
** Under the License Agreement, this software may be used solely to create
** custom applications for use on the Lantronix xPico Wi-Fi product.
** THIS SOFTWARE AND ANY ACCOMPANYING DOCUMENTATION IS PROVIDED "AS IS".
** LANTRONIX SPECIFICALLY DISCLAIMS ANY WARRANTIES, INCLUDING, BUT NOT LIMITED
** TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY, NON-INFRINGEMENT AND FITNESS
** FOR A PARTICULAR PURPOSE.
** LANTRONIX HAS NO OBLIGATION TO PROVIDE MAINTENANCE, SUPPORT, UPDATES,
** ENHANCEMENTS, OR MODIFICATIONS TO THIS SOFTWARE.
** IN NO EVENT SHALL LANTRONIX BE LIABLE TO ANY PARTY FOR DIRECT, INDIRECT,
** SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES, INCLUDING LOST PROFITS,
** ARISING OUT OF THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF
** LANTRONIX HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
*/


/****************************************************************************/
/*                                 Includes                                 */
/****************************************************************************/

#include <inttypes.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include "ble_explorer_module_defs.h"
#include "ble_strings.h"
#include "ble_utils.h"
#include "bluetooth_module_libs.h"
#include "ltrx_cfgvar.h"
#include "ltrx_definitions.h"
#include "ltrx_bluetooth.h"
#include "ltrx_gatt.h"
#include "ltrx_stream.h"
#include "ltrx_tlog.h"
#include "ltrx_xml.h"
#include "uuid.h"


/****************************************************************************/
/*                                  Macros                                  */
/****************************************************************************/

/* Don't need (or want) ctype isspace() */
#define _is_space(_c) ((_c) == ' ' || (_c) == '\t')

#define _is_quote(_c) ((_c) == '"' || (_c) == '\'')


/****************************************************************************/
/*                           Function Definitions                           */
/****************************************************************************/

void stream_printf(
    struct output_stream *os,
    const char *format,
    ...
)
{
    va_list args;
    va_start(args, format);
    char *s = ltrx_vsprintf_malloc(format, args);
    va_end(args);
    if (s)
    {
        ltrx_output_stream_write_without_ending_line(os, s);
        ltrx_free(s);
    }
}

bool str_bdaddr(const char *strbuffer, uint8_t bda[])
{
    char address[18];
    memset(address, 0, sizeof(address));
    memset(bda, 0, BD_ADDRESS_SIZE);
    strlcpy(address, strbuffer, sizeof(address));
    char *p = strtok(address, " :-");
    uint8_t nsections = 1;
    if (p != NULL)
    {
        bda[0] = strtoul(p, NULL, 16);
        for ( ; p != NULL && nsections < BD_ADDRESS_SIZE; ++nsections)
        {
            p = strtok(NULL, " :-");
            if (p != NULL)
            {
                bda[nsections] = strtoul(p, NULL, 16);
            }
        }
    }
    return nsections == BD_ADDRESS_SIZE && BD_ADDR_IS_VALID(bda);
}

void bt_uuid_parse(const char *s, struct ltrx_uuid *lu)
{
    memset(lu, 0, sizeof(*lu));
    if (strlen(s) >= UUID_STRING_LENGTH)
    {
        uuid_t u;
        uuid_parse(s, u);
        lu->length = UUID_LENGTH_128;
        _reverse_byte_array(u, sizeof(u));
        memcpy(lu->uu.id128, u, UUID_LENGTH_128);
    }
    else
    {
        lu->uu.id32 = strtoul(s, NULL, 0);
        lu->length = (lu->uu.id32 > 0xFFFF) ? UUID_LENGTH_32 : UUID_LENGTH_16;
    }
}

char *bt_uuid_unparse(struct ltrx_uuid *lu, char *buffer, size_t length)
{
    static char default_buffer[40];
    if (buffer == NULL)
    {
        buffer = default_buffer;
        length = sizeof(default_buffer);
    }
    if (lu->length == UUID_LENGTH_16)
    {
        snprintf(buffer, length, "0x%04X", lu->uu.id16);
    }
    else if (lu->length == UUID_LENGTH_32)
    {
        snprintf(buffer, length, "0x%08" PRIx32 "", lu->uu.id32);
    }
    else
    {
        uuid_t u;
        memcpy(u, lu->uu.id128, sizeof(u));
        _reverse_byte_array(u, sizeof(u));
        uuid_unparse(u, buffer);
    }
    return buffer;
}

bool buffer_arg_init(
    const char *buffer, struct buffer_arg_context *bac, uint32_t *argcountp
)
{
    memset(bac, 0, sizeof(*bac));
    bac->buffer = STRDUP(buffer);
    if (! bac->buffer)
    {
        return false;
    }
    bac->length = strlen(buffer) + 1;
    if (argcountp)
    {
        if (bac->length <= 1)
        {
            *argcountp = 0;
        }
        else
        {
            uint32_t argc = 0;
            while (_is_space(bac->buffer[bac->current]))
            {
                bac->current++;
            }
            if (_is_quote(bac->buffer[bac->current]))
            {
                bac->inquotes = true;
                bac->current++;
                argc++;
            }
            while (bac->buffer[bac->current])
            {
                if (bac->inquotes)
                {
                    if (_is_quote(bac->buffer[bac->current]))
                    {
                        bac->inquotes = false;
                        bac->current++;
                        argc++;
                    }
                }
                else
                {
                    if (_is_space(bac->buffer[bac->current]))
                    {
                        bac->current++;
                        argc++;
                    }
                }
                bac->current++;
            }
            if (! _is_space(bac->buffer[bac->current - 1]))
            {
                argc++;
            }
            *argcountp = argc;
        }
    }

    bac->current = 0;
    bac->inquotes = false;
    while (_is_space(bac->buffer[bac->current]))
    {
        bac->current++;
    }
    bac->last = bac->current;
    return true;
}

void buffer_arg_term(struct buffer_arg_context *bac)
{
    ltrx_free(bac->buffer);
}

char *buffer_arg_next(struct buffer_arg_context *bac)
{
    char *p;
    while (_is_space(bac->buffer[bac->current]))
    {
        bac->current++;
    }
    if (_is_quote(bac->buffer[bac->current]))
    {
        bac->inquotes = true;
        bac->current++;
    }
    bac->last = bac->current;
    p = &bac->buffer[bac->last];

    while (bac->current < bac->length)
    {
        if (bac->inquotes)
        {
            if (_is_quote(bac->buffer[bac->current]))
            {
                bac->inquotes = false;
                bac->buffer[bac->current] = 0;
                if (bac->current < bac->length)
                {
                    bac->current++;
                }
                break;
            }
        }
        else if (_is_space(bac->buffer[bac->current]))
        {
            bac->buffer[bac->current] = 0;
            if (bac->current < bac->length)
            {
                bac->current++;
            }
            break;
        }
        else if (! bac->buffer[bac->current])
        {
            if (bac->current < bac->length)
            {
                bac->current++;
            }
            break;
        }
        bac->current++;
    }
    return p;
}

int buffer_arg_getopt(
    struct buffer_arg_context *bac,
    const char *options,
    char **optargp
)
{
    if (optargp)
    {
        *optargp = NULL;
    }
    bool argfound = false;
    do {
        char *p = buffer_arg_next(bac);

        if (*p == '-')
        {
            argfound = true;
            p++;
            for (uint8_t i = 0; i < strlen(options); ++i)
            {
                if (options[i] == *p)
                {
                    p++;
                    if (options[i + 1] == ':' && optargp)
                    {
                        *optargp = buffer_arg_next(bac);
                    }
                    return (int)options[i];
                }
            }
        }

    } while (bac->buffer[bac->current] == '-');

    if (! argfound)
    {
        bac->current = bac->last;
    }
    return -1;
}

