#
# Copyright 2015, Broadcom Corporation
# All Rights Reserved.
#
# This is UNPUBLISHED PROPRIETARY SOURCE CODE of Broadcom Corporation;
# the contents of this file may not be disclosed to third parties, copied
# or duplicated in any form, in whole or in part, without the prior
# written permission of Broadcom Corporation.
#
# Modifications Copyright 2016-2023, Lantronix Inc.
#
###############################################################################
#
# This TCL OpenOCD script runs on a PC and communicates with the embedded
# sflash_write app to allow the PC to write data into a serial flash chip
# attached to the target processor
#
# Usage example:
#
# source [find mfg_spi_flash/write_sflash.tcl]
# sflash_init "BCMUSI11-SDIO-debug"
# sflash_write_file "example.bin" 0x10 "BCMUSI11-SDIO-debug" 1
# shutdown
#
###############################################################################

# CHIP_RAM_START must be supplied by target specific TCL script
set MemoryStart $CHIP_RAM_START

###############################################################################
#
# These variables must match the ones in Apps/waf/sflash_write/sflash_write.c
#
# They rely on the linker script placing the data_config_area_t and
# data_transfer_area_t structures at the start of RAM
#
###############################################################################

# This must match data_config_area_t
set entry_address_loc   [expr $MemoryStart + 0x00 ]
set stack_address_loc   [expr $MemoryStart + 0x04 ]
set buffer_size_loc     [expr $MemoryStart + 0x08 ]

# This must match data_transfer_area_t
set log_port            [expr $MemoryStart + 0x0C ]
set log_level_loc       [expr $MemoryStart + 0x10 ]
set data_size_loc       [expr $MemoryStart + 0x14 ]
set dest_address_loc    [expr $MemoryStart + 0x18 ]
set command_loc         [expr $MemoryStart + 0x1C ]
set result_loc          [expr $MemoryStart + 0x20 ]
set data_loc            [expr $MemoryStart + 0x24 ]


# These must match the MFG_SPI_FLASH_COMMAND defines
set COMMAND_INITIAL_VERIFY            (0x01)
set COMMAND_ERASE                     (0x02)
set COMMAND_WRITE                     (0x04)
set COMMAND_POST_WRITE_VERIFY         (0x08)
set COMMAND_VERIFY_CHIP_ERASURE       (0x10)
set COMMAND_READ                      (0x40)
set COMMAND_WRITE_ERASE_IF_NEEDED     (0x80)

# These must match the mfg_spi_flash_result_t enum
set RESULT(0xffffffff)      "In Progress"
set RESULT(4294967295)      "In Progress"
set RESULT(0)               "OK"
set RESULT(1)               "Erase Failed"
set RESULT(2)               "Verify after write failed"
set RESULT(3)               "Size too big for buffer"
set RESULT(4)               "Size too big for chip"
set RESULT(5)               "DCT location not found - has factory reset app been written?"
set RESULT(6)               "Error during Write"
set RESULT(7)               "Error during Read"

###############################################################################
# memread32
#
# Helper function that reads a 32 bit value from RAM and returns it
#
# address - the RAM address to read from
###############################################################################
proc memread32 {address resume_required} {
    if { $resume_required == 1 } {
        halt
    }
    mem2array memar 32 $address 1
    if { $resume_required == 1} {
        resume
    }
    return $memar(0)
}

###############################################################################
# load_image_bin
#
# Loads part of a binary file into RAM
#
# fname   - filename of binary image file
# foffset - offset from the start of the binary file where data will be read
# address - the destination RAM address
# length  - number of bytes to transfer
###############################################################################
proc load_image_bin {fname foffset address length } {

    # Load data from fname filename at foffset offset to
    # target at address. Load at most length bytes.
    puts [format "loadimage address 0x%8.8X foffset $foffset $length" $address]
    load_image $fname [expr $address - $foffset] bin $address $length
}

###############################################################################
# sflash_init
#
# Prepares for writing to serial flashby loading the sflash_write app into
# memory and setting it running.
# This function assumes the following target has already been built:
#      waf.sflash_write-NoOS-<PlatBusDebug>
#
# PlatBusDebug   - The platform, bus and debug part of the build target
###############################################################################
proc sflash_init { flashWriteApp { logLevel 0 } } {
    global entry_address_loc
    global stack_address_loc
    global buffer_size_loc
    global log_level_loc
    global entry_address
    global stack_address
    global buffer_size


    puts "sflash_init with log level $logLevel"

    puts "==== init ========================================================="
    init

    gdb_breakpoint_override hard

    puts "==== reset halt ==================================================="
    sleep 100
    reset halt

    cortex_r4 maskisr on
    adapter_khz 10000

    # Go to Supervisor ARM mode
    reg cpsr 0xd3
    reg lr   0x0
    halt 2000

    load_image $flashWriteApp

    set entry_address [memread32 $entry_address_loc 0]
    set stack_address [memread32 $stack_address_loc 0]
    set buffer_size   [memread32 $buffer_size_loc 0]

    # new ejl
    #puts [format "log_level @ 0x%8.8X  level=$logLevel" $log_level_loc]
    mww $log_level_loc $logLevel

    puts [format "entry_address= 0x%8.8X" $entry_address]
    puts [format "stack_address= 0x%8.8X" $stack_address]
    puts "buffer_size= $buffer_size"
    if { $buffer_size == 0 } {
        puts "Error: Buffer size read from address $buffer_size_loc on target is zero"
        exit -1;
    }

    # Setup start address
    reg pc $entry_address

    resume
}

###############################################################################
# program_sflash
#
# Executes a serial command by communicating to the sflash_write app
#
# fname    - filename of binary image file (if command requires it)
# foffset  - offset from the start of the binary file where data will be read  (if command requires it)
# dataSize - number of bytes to transfer (if command requires it)
# destaddr - the destination serial flash address (if command requires it)
# cmd      - The commmand to execute (see list above)
###############################################################################
proc program_sflash { filename foffset dataSize destaddr cmd } {
    global MemoryStart
    global data_size_loc
    global data_loc
    global dest_address_loc
    global command_loc
    global result_loc
    global entry_address
    global RESULT
    global entry_address_loc

    halt
    # Load the binary data into the RAM
    if { ( $dataSize != 0 ) && ( $filename != "" ) } {
       load_image_bin $filename $foffset $data_loc $dataSize
    }

    # Write the details of the data
    mww $data_size_loc    $dataSize
    mww $dest_address_loc $destaddr
    mww $result_loc       0xffffffff

    #mdw $entry_address_loc 100

    # Write the command - This causes the writing to start
    mww $command_loc      $cmd
    resume

    set resultval 0xffffffff
    set loops  0
    puts "==== poll loop ===================================================="
    while { ($resultval == 0xffffffff) && ( $loops < 500 ) } {
        sleep 250
        set resultval [memread32 $result_loc 1]
        incr loops
    }
    puts "==== res=$resultval loops=$loops"

    if { $resultval != 0 } {
        puts "Error: $resultval"
        halt
        reg
        exit -1;
    }
}

###############################################################################
# sflash_write_file
#
# Writes an entire binary image file to a serial flash address
# This function assumes the following target has already been built:
#      waf.sflash_write-NoOS-<PlatBusDebug>
#
# filename     - filename of binary image file
# destAddress  - the destination serial flash address
###############################################################################
proc sflash_write_file { flashWriteApp filename destAddress { logLevel 0 } } {
    global COMMAND_ERASE 
    global COMMAND_INITIAL_VERIFY 
    global COMMAND_WRITE 
    global COMMAND_POST_WRITE_VERIFY 
    global buffer_size 
    global COMMAND_WRITE_ERASE_IF_NEEDED

    puts "sflash_write_file with loglevel $logLevel"
    sflash_init $flashWriteApp $logLevel

    set binDataSize [file size $filename]
    set write_command_val [expr $COMMAND_WRITE_ERASE_IF_NEEDED | $COMMAND_POST_WRITE_VERIFY ]
    set pos 0
    puts "Total write size is $binDataSize"
    while { $pos < $binDataSize } {
        if { ($binDataSize - $pos) <  $buffer_size } {
            set writesize [expr ($binDataSize - $pos)]
        } else {
            set writesize $buffer_size
        }
        puts [format "  writing $writesize bytes at 0x%8.8X" [expr $destAddress + $pos]]
        program_sflash $filename $pos $writesize [expr $destAddress + $pos] $write_command_val
        set pos [expr $pos + $writesize]
    }
}

###############################################################################
# sflash_read_file
#
# Reads data from a serial flash address
# This function assumes the following target has already been built:
#      waf.sflash_write-NoOS-<PlatBusDebug>
#
# filename     - output filename for binary image file
# srcAddress   - the destination serial flash address
# length       - number of bytes to read
###############################################################################
proc sflash_read_file { flashWriteApp filename srcAddress length } {
    global COMMAND_ERASE 
    global COMMAND_INITIAL_VERIFY 
    global COMMAND_WRITE 
    global COMMAND_POST_WRITE_VERIFY 
    global buffer_size 
    global COMMAND_READ 
    global data_loc

    sflash_init $flashWriteApp
    set temp_file "temp.bin"

    set read_command_val [expr $COMMAND_READ ]
    set pos 0

    puts "Total read size is $length"
    while { $pos < $length } {
        if { ($length - $pos) <  $buffer_size } {
            set readsize [expr ($length - $pos)]
        } else {
            set readsize $buffer_size
        }
        puts "  reading $readsize bytes from [expr $srcAddress + $pos]"
        program_sflash "" $pos $readsize [expr $srcAddress + $pos] $read_command_val
        puts "  dumping image from $data_loc $readsize"
        halt
        dump_image  $temp_file $data_loc $readsize

        # There is apparently no way to keep windows from inserting a
        # 0x0D when it sees a 0x0A, because all tools (cat, sed, ..?) 
        # open the file in Windows' "text mode"
        #exec cat $temp_file >> $filename
        exec sed '' -b $temp_file >>$filename
        set pos [expr $pos + $readsize]
    }
}

proc sflash_erase { flashWriteApp { logLevel 0 } } {
    global COMMAND_ERASE 
    global COMMAND_INITIAL_VERIFY 
    global COMMAND_WRITE 
    global COMMAND_POST_WRITE_VERIFY 
    global buffer_size 
    global COMMAND_WRITE_ERASE_IF_NEEDED

    sflash_init $flashWriteApp $logLevel

    set erase_command_val [expr $COMMAND_ERASE ]

    puts "Erasing Chip"
    program_sflash "" 0 0 0 $erase_command_val

#    set pos 0
#    while { $pos < $length } {
#        if { ($length - $pos) <  $buffer_size } {
#            set size [expr ($length - $pos)]
#        } else {
#            set size $buffer_size
#        }
#        puts "  erasing @ [expr $start + $pos] size $size"
#        program_sflash "" 0 $size [expr $start + $pos] $erase_command_val
#        set pos [expr $pos + $size]
#    }
    puts "Chip Erase Done"
}

