
/*****************************************************************************/
/*                              Legal                                        */
/*****************************************************************************/

/*
** Copyright ©2015-2025, Lantronix, Inc. All Rights Reserved.
** By using this software, you are agreeing to the terms of the Software
** Development Kit (SDK) License Agreement included in the distribution package
** for this software (the “License Agreement”).
** Under the License Agreement, this software may be used solely to create
** custom applications for use on the Lantronix xPico Wi-Fi, xPico 200 Series,
** and xPort® Edge products.
** THIS SOFTWARE AND ANY ACCOMPANYING DOCUMENTATION IS PROVIDED "AS IS".
** LANTRONIX SPECIFICALLY DISCLAIMS ANY WARRANTIES, INCLUDING, BUT NOT LIMITED
** TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY, NON-INFRINGEMENT AND FITNESS
** FOR A PARTICULAR PURPOSE.
** LANTRONIX HAS NO OBLIGATION TO PROVIDE MAINTENANCE, SUPPORT, UPDATES,
** ENHANCEMENTS, OR MODIFICATIONS TO THIS SOFTWARE.
** IN NO EVENT SHALL LANTRONIX BE LIABLE TO ANY PARTY FOR DIRECT, INDIRECT,
** SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES, INCLUDING LOST PROFITS,
** ARISING OUT OF THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF
** LANTRONIX HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
*/

/*****************************************************************************/
/*                             Documentation                                  */
/*****************************************************************************/

/*!
** \addtogroup example
** @{
*/

/*!
** \defgroup secure_http_client secure_http_client
** @{
**
** The \b secure_http_client module demonstrates using the HTTP Client to
** create a POST transaction to a website, and print the response to the
** serial port.
**
** To use it:
**  1. Create a credential with the name httpbin
**  2. Find the root certificate presented by httpbin.org with openssl:
**     openssl s_client -showcerts -connect httpbin.org:443
**     - Copy and past the last certificate in the chain (the root) including
**       the lines --BEGIN CERTIFICATE-- and --END CERTIFICATE-- into the
**       Certificate box of the httpbin credential created in step 1
**     - Configure Line 1 to Protocol HTTPS Client
**
** An alternative approach, if you do not want to install a certificate
** (say, to avoid configuring multiple devices), is to use the certificate
** authority directly. This technique can be used by disabling the
** USE_INSTALLED_CERTIFICATE define.
**
**  Built it from project "secureHttpClientDemo"
*/


/*****************************************************************************/
/*                             Includes                                      */
/*****************************************************************************/

#include <stdio.h>  /* Delivered with SDK. */
#include <string.h> /* Delivered with SDK. */

#include "secure_http_client_module_defs.h" /* Automatically generated by make. */
#include "security_module_libs.h"           /* Bring in the security module's functions, for TLS */

#include "ltrx_stream.h"          /* Delivered with SDK. */
#include "ltrx_tlog.h"            /* Delivered with SDK. */
#include "ltrx_line.h"            /* Delivered with SDK. */
#include "ltrx_network.h"
#include "ltrx_http_client.h"

/*****************************************************************************/
/*                              Defines                                      */
/*****************************************************************************/

#define USE_INSTALLED_CERTIFICATE

//#define DEBUG

/*****************************************************************************/
/*                              Globals                                      */
/*****************************************************************************/

const struct security_external_functions *g_securityExternalFunctionEntry_pointer;

/*****************************************************************************/
/*                             Structs                                       */
/*****************************************************************************/

struct thread_info
{
    uint32_t zeroBasedIndex;
    bool isRunning;
    struct output_stream_to_uart ostu;
};

/*****************************************************************************/
/*                            Prototypes                                     */
/*****************************************************************************/

bool StartLineProtocol(uint16_t zeroBasedIndex);

void StopLineProtocol(uint16_t zeroBasedIndex);

/*****************************************************************************/
/*                         Local Constants                                   */
/*****************************************************************************/

static const struct ltrx_line_protocol s_lineProtocol =
{
    .protocolName = "HTTPS Client",
    .helpHtml = "SDK example.",
    .startProtocol = StartLineProtocol,
    .stopProtocol = StopLineProtocol
};

/*****************************************************************************/
/*                         Local Variables                                   */
/*****************************************************************************/

static struct thread_info *s_threadInfo[MAX_LOGICAL_SERIAL_LINES];

static struct ltrx_thread *s_threadForLine[MAX_LOGICAL_SERIAL_LINES];

#ifdef USE_INSTALLED_CERTIFICATE
static const char s_installedCredential[] = "httpbin";
#else
static const char s_trustedAuthority[] = 
    "-----BEGIN CERTIFICATE-----\n"
    "MIIEdTCCA12gAwIBAgIJAKcOSkw0grd/MA0GCSqGSIb3DQEBCwUAMGgxCzAJBgNV\n"
    "BAYTAlVTMSUwIwYDVQQKExxTdGFyZmllbGQgVGVjaG5vbG9naWVzLCBJbmMuMTIw\n"
    "MAYDVQQLEylTdGFyZmllbGQgQ2xhc3MgMiBDZXJ0aWZpY2F0aW9uIEF1dGhvcml0\n"
    "eTAeFw0wOTA5MDIwMDAwMDBaFw0zNDA2MjgxNzM5MTZaMIGYMQswCQYDVQQGEwJV\n"
    "UzEQMA4GA1UECBMHQXJpem9uYTETMBEGA1UEBxMKU2NvdHRzZGFsZTElMCMGA1UE\n"
    "ChMcU3RhcmZpZWxkIFRlY2hub2xvZ2llcywgSW5jLjE7MDkGA1UEAxMyU3RhcmZp\n"
    "ZWxkIFNlcnZpY2VzIFJvb3QgQ2VydGlmaWNhdGUgQXV0aG9yaXR5IC0gRzIwggEi\n"
    "MA0GCSqGSIb3DQEBAQUAA4IBDwAwggEKAoIBAQDVDDrEKvlO4vW+GZdfjohTsR8/\n"
    "y8+fIBNtKTrID30892t2OGPZNmCom15cAICyL1l/9of5JUOG52kbUpqQ4XHj2C0N\n"
    "Tm/2yEnZtvMaVq4rtnQU68/7JuMauh2WLmo7WJSJR1b/JaCTcFOD2oR0FMNnngRo\n"
    "Ot+OQFodSk7PQ5E751bWAHDLUu57fa4657wx+UX2wmDPE1kCK4DMNEffud6QZW0C\n"
    "zyyRpqbn3oUYSXxmTqM6bam17jQuug0DuDPfR+uxa40l2ZvOgdFFRjKWcIfeAg5J\n"
    "Q4W2bHO7ZOphQazJ1FTfhy/HIrImzJ9ZVGif/L4qL8RVHHVAYBeFAlU5i38FAgMB\n"
    "AAGjgfAwge0wDwYDVR0TAQH/BAUwAwEB/zAOBgNVHQ8BAf8EBAMCAYYwHQYDVR0O\n"
    "BBYEFJxfAN+qAdcwKziIorhtSpzyEZGDMB8GA1UdIwQYMBaAFL9ft9HO3R+G9FtV\n"
    "rNzXEMIOqYjnME8GCCsGAQUFBwEBBEMwQTAcBggrBgEFBQcwAYYQaHR0cDovL28u\n"
    "c3MyLnVzLzAhBggrBgEFBQcwAoYVaHR0cDovL3guc3MyLnVzL3guY2VyMCYGA1Ud\n"
    "HwQfMB0wG6AZoBeGFWh0dHA6Ly9zLnNzMi51cy9yLmNybDARBgNVHSAECjAIMAYG\n"
    "BFUdIAAwDQYJKoZIhvcNAQELBQADggEBACMd44pXyn3pF3lM8R5V/cxTbj5HD9/G\n"
    "VfKyBDbtgB9TxF00KGu+x1X8Z+rLP3+QsjPNG1gQggL4+C/1E2DUBc7xgQjB3ad1\n"
    "l08YuW3e95ORCLp+QCztweq7dp4zBncdDQh/U90bZKuCJ/Fp1U1ervShw3WnWEQt\n"
    "8jxwmKy6abaVd38PMV4s/KCHOkdp8Hlf9BRUpJVeEXgSYCfOn8J3/yNTd126/+pZ\n"
    "59vPr5KW7ySaNRB6nJHGDn2Z9j8Z3/VyVOEVqQdZe4O/Ui5GjLIAZHYcSNPYeehu\n"
    "VsyuLAOQ1xk4meTKCRlb/weWsKh/NEnfVqn3sF/tM+2MR7cwA130A4w=\n"
    "-----END CERTIFICATE-----";
#endif


/*****************************************************************************/
/*                               Code                                        */
/*****************************************************************************/

static bool http_response_callback(
    struct ltrx_http_client_request *lhcr,
    void *arg,
    const void *data,
    size_t len)
{
    (void)lhcr;
    struct thread_info *ti = (struct thread_info *)arg;

    static uint32_t received;
    uint32_t length;
    ltrx_httpc_request_get_content_length(lhcr, &length);
    ltrx_output_stream_write_binary(&ti->ostu.outStream, data, len);
    received += len;

    if (received >= length)
    {
        received = 0;
        return true;
    }
    return false;
}

static int send_http_post(
    struct thread_info *ti,
    const char *host,
    const char *url,
    const char *buffer
)
{
    struct ltrx_http_client_connect_info lhcci = {
        .host = host,
        .url = url,
        .host_port = 443
    };

    struct ltrx_http_client_request *lhcr = ltrx_httpc_request_create(LTRX_HTTPC_METHOD__POST);
    if (! lhcr)
    {
        TLOG(TLOG_SEVERITY_LEVEL__ERROR, "Failed to create HTTP request");
        return -1;
    }
    if (ltrx_httpc_request_set_connect_info(lhcr, &lhcci) != 0)
    {
        TLOG(TLOG_SEVERITY_LEVEL__ERROR, "Failed to set HTTP connect info");
        goto error;
    }
    struct ltrx_http_client_header hdr = {
        .name = "Content-Type",
        .value = "application/json"
    };
    if (ltrx_httpc_request_add_header(lhcr, &hdr) != 0)
    {
        TLOG(TLOG_SEVERITY_LEVEL__ERROR, "Failed to add request header");
        goto error;
    }
#ifdef USE_INSTALLED_CERTIFICATE
    if (ltrx_httpc_request_set_tls_credential(lhcr, s_installedCredential) != 0)
    {
        TLOG(TLOG_SEVERITY_LEVEL__ERROR, "Failed to set TLS credential");
        goto error;
    }
#else
    if (ltrx_httpc_request_add_tls_authority(lhcr, s_trustedAuthority) != 0)
    {
        TLOG(TLOG_SEVERITY_LEVEL__ERROR, "Failed to add CA");
        goto error;
    }
#endif

    if (ltrx_httpc_request_set_write_buffer(lhcr, buffer, strlen(buffer)) != 0)
    {
        TLOG(TLOG_SEVERITY_LEVEL__ERROR, "Failed to set write buffer");
        goto error;
    }
    if (ltrx_httpc_request_set_response_callback(lhcr, http_response_callback, ti) != 0)
    {
        TLOG(TLOG_SEVERITY_LEVEL__ERROR, "Failed to set response callback");
        goto error;
    }
#ifdef DEBUG
    ltrx_httpc_request_set_debug(lhcr, true, true);
#endif
    if (ltrx_httpc_request_execute(lhcr) != 0)
    {
        TLOG(TLOG_SEVERITY_LEVEL__ERROR, "Failed to execute HTTP request");
        goto error;
    }
    return 0;

error:
    ltrx_httpc_request_destroy(lhcr);
    return -1;
}

static void lineLoop(struct thread_info *ti)
{
    ltrx_output_stream_init_to_uart(&ti->ostu, ti->zeroBasedIndex);
    ltrx_output_stream_write_line(&ti->ostu.outStream, "");
    while (ti->isRunning)
    {
#ifdef USE_INSTALLED_CERTIFICATE
        if (! ltrx_ssl_credential_is_present(s_installedCredential))
        {
            bool logged = false;
            if (! logged)
            {
                TLOG(TLOG_SEVERITY_LEVEL__ERROR, "TLS Credential not found");
                logged = true;
            }
            continue;
        }
#endif
        send_http_post(
            ti,
            "httpbin.org",
            "/post",
            "{\"temperature\": 25, \"name\": \"widget\"}"
        );
        ltrx_thread_sleep(10000);
    }
}

static void lineThread(void *opaque)
{
    uint16_t zeroBasedIndex = (uint32_t)opaque;
    uint16_t line = zeroBasedIndex + 1;
    bool loggedStartMessage = false;
    struct thread_info ti = {
        .zeroBasedIndex = zeroBasedIndex,
        .isRunning = true
    };
    s_threadInfo[zeroBasedIndex] = &ti;
    while (
        ti.isRunning &&
        ! ltrx_line_open(zeroBasedIndex, 1000)
    )
    {
    }
    if (ti.isRunning)
    {
        TLOG(
            TLOG_SEVERITY_LEVEL__INFORMATIONAL,
            "%s started on Line %u",
            s_lineProtocol.protocolName,
            line
        );
        loggedStartMessage = true;
        ltrx_line_set_dtr(zeroBasedIndex, true);
        lineLoop(&ti);
    }
    if (loggedStartMessage)
    {
        TLOG(
            TLOG_SEVERITY_LEVEL__INFORMATIONAL,
            "%s stopped on Line %u",
            s_lineProtocol.protocolName,
            line
        );
    }
    ltrx_line_close(zeroBasedIndex);
    s_threadInfo[zeroBasedIndex] = NULL;
    s_threadForLine[zeroBasedIndex] = 0;
}

bool StartLineProtocol(uint16_t zeroBasedIndex)
{
    uint16_t line = zeroBasedIndex + 1;
    if (s_threadInfo[zeroBasedIndex] || s_threadForLine[zeroBasedIndex])
    {
        TLOG(
            TLOG_SEVERITY_LEVEL__ERROR,
            "%s thread already running for Line %u",
            s_lineProtocol.protocolName,
            line
        );
        return false;
    }
    g_securityExternalFunctionEntry_pointer = ltrx_module_functions_lookup(
        "Security"
    );

    if (! g_securityExternalFunctionEntry_pointer)
    {
        TLOG(TLOG_SEVERITY_LEVEL__ERROR, "Security module not found");
    }
    else
    {
        s_threadForLine[zeroBasedIndex] = ltrx_thread_create(
            s_lineProtocol.protocolName,
            lineThread,
            (void *)(uint32_t)zeroBasedIndex,
            5000
        );
    }

    if (! s_threadForLine[zeroBasedIndex])
    {
        TLOG(
            TLOG_SEVERITY_LEVEL__ERROR,
            "Failed to create %s thread for Line %u",
            s_lineProtocol.protocolName,
            line);
        return false;
    }
    return true;
}

void StopLineProtocol(uint16_t zeroBasedIndex)
{
    bool wasRunning = false;
    if(zeroBasedIndex < MAX_LOGICAL_SERIAL_LINES)
    {
        ltrx_preemption_block();
        struct thread_info *ti = s_threadInfo[zeroBasedIndex];
        if(ti && ti->isRunning)
        {
            wasRunning = true;
            ti->isRunning = false;
        }
        ltrx_preemption_unblock();
        if(wasRunning)
        {
            struct ltrx_thread *lt;
            uint32_t tm = ltrx_timemark();
            while(
                (lt = s_threadForLine[zeroBasedIndex]) != NULL &&
                lt != ltrx_thread_id() &&
                ltrx_elapsed_time_current_ms(tm) < 2000
            )
            {
                if(ltrx_elapsed_time_current_ms(tm) >= 500)
                {
                    ltrx_line_purge(zeroBasedIndex);
                }
                ltrx_thread_sleep(100);
            }
        }
    }
}

void secure_http_client_module_registration(void)
{
    ltrx_module_register(&g_secure_http_clientModuleInfo);
    ltrx_line_register_protocol(&s_lineProtocol);
}

void secure_http_client_module_startup(void)
{
}

void secure_http_client_module_shutdown(void)
{
}

/*!
** @}
*/

/*!
** @}
*/
