
/*****************************************************************************/
/*                              Legal                                        */
/*****************************************************************************/

/*
** Copyright 2015-2025, Lantronix, Inc. All Rights Reserved.
** By using this software, you are agreeing to the terms of the Software
** Development Kit (SDK) License Agreement included in the distribution package
** for this software (the License Agreement).
** Under the License Agreement, this software may be used solely to create
** custom applications for use on the Lantronix xPico Wi-Fi product.
** THIS SOFTWARE AND ANY ACCOMPANYING DOCUMENTATION IS PROVIDED "AS IS".
** LANTRONIX SPECIFICALLY DISCLAIMS ANY WARRANTIES, INCLUDING, BUT NOT LIMITED
** TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY, NON-INFRINGEMENT AND FITNESS
** FOR A PARTICULAR PURPOSE.
** LANTRONIX HAS NO OBLIGATION TO PROVIDE MAINTENANCE, SUPPORT, UPDATES,
** ENHANCEMENTS, OR MODIFICATIONS TO THIS SOFTWARE.
** IN NO EVENT SHALL LANTRONIX BE LIABLE TO ANY PARTY FOR DIRECT, INDIRECT,
** SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES, INCLUDING LOST PROFITS,
** ARISING OUT OF THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF
** LANTRONIX HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
*/

/*****************************************************************************/
/*                           Documentation                                   */
/*****************************************************************************/

/*!
** \addtogroup example
** @{
*/

/*!
** \defgroup xml_access xml_access
** @{
**
** The \b xml_access module demonstrates how to read device status and
** configuration values via XML.
**
** Build it from project "xmlAccessDemo".
**
** Note that results are sent to the log via a character buffer.
** If you were sending to a Line or network interface, you could instead stream
** the result without requiring a buffer.
*/

/*****************************************************************************/
/*                             Includes                                      */
/*****************************************************************************/

#include "ltrx_network.h" /* Delivered with SDK. */
#include "ltrx_stream.h" /* Delivered with SDK. */
#include "ltrx_tlog.h" /* Delivered with SDK. */
#include "xml_access_module_defs.h" /* Automatically generated by make. */

/*****************************************************************************/
/*                         Local Constants                                   */
/*****************************************************************************/

static const uint32_t s_delayTimeInMilliseconds = 60000;

static const uint8_t s_zeroBasedInterface = NETS_WLAN_START;

/*****************************************************************************/
/*                               Code                                        */
/*****************************************************************************/

static bool getXmlValue(
    const struct xml_emit_value_specification *xevs,
    char *buffer,
    size_t size
)
{
    struct output_stream_to_buffer osb;
    ltrx_output_stream_init_to_buffer(
		&osb, buffer, size, OUTPUT_STREAM_TO_BUFFER_MODE__ZERO_TERMINATE
	);
    return(ltrx_xml_emit_value(xevs, &osb.outStream));
}

static void xmlAccessThread(void *opaque)
{
    (void)opaque;
    char buffer[50];
    while(true)
    {
        /* Firmware version. */
        {
	        static const struct xml_emit_value_specification s_xevs =
            {
		        .type = XML_EMIT_VALUE_TYPE__STATUS,
		        .groupName = "Device",
		        .itemName = "Firmware Version"
	        };
            if(getXmlValue(&s_xevs, buffer, sizeof(buffer)))
            {
                TLOG(
                    TLOG_SEVERITY_LEVEL__INFORMATIONAL,
                    "Firmware version: %s", buffer
                );
            }
        }
        /* MAC address. */
        {
	        static const struct xml_emit_value_specification s_xevs =
            {
		        .type = XML_EMIT_VALUE_TYPE__STATUS,
		        .groupName = "Interface",
		        .optGroupInstance = "wlan0",
		        .itemName = "MAC Address"
	        };
            if(getXmlValue(&s_xevs, buffer, sizeof(buffer)))
            {
                TLOG(
                    TLOG_SEVERITY_LEVEL__INFORMATIONAL,
                    "MAC address: %s", buffer
                );
            }
        }
        /* Current wlan0 IP address. */
        if(ltrx_ip_address_state(s_zeroBasedInterface) > 0)
        {
	        static const struct xml_emit_value_specification s_xevs =
            {
		        .type = XML_EMIT_VALUE_TYPE__STATUS,
		        .groupName = "Interface",
		        .optGroupInstance = "wlan0",
		        .itemName = "Current",
		        .optValueName = "IP Address"
	        };
            if(getXmlValue(&s_xevs, buffer, sizeof(buffer)))
            {
                TLOG(
                    TLOG_SEVERITY_LEVEL__INFORMATIONAL,
                    "wlan0 IP address: %s", buffer
                );
            }
        }
#ifdef USE_POWER_MANAGEMENT
        /* Configured dynamic power mode. */
        {
	        static const struct xml_emit_value_specification s_xevs =
            {
		        .type = XML_EMIT_VALUE_TYPE__CONFIGURATION,
		        .groupName = "Power",
		        .itemName = "Dynamic Power Mode"
	        };
            if(getXmlValue(&s_xevs, buffer, sizeof(buffer)))
            {
                TLOG(
                    TLOG_SEVERITY_LEVEL__INFORMATIONAL,
                    "Dynamic power mode: %s", buffer
                );
            }
        }
#endif
        ltrx_thread_sleep(s_delayTimeInMilliseconds);
    }
}

void xml_access_module_registration(void)
{
    ltrx_module_register(&g_xml_accessModuleInfo);
}

void xml_access_module_startup(void)
{
    ltrx_thread_create(
        "XML Access",
        xmlAccessThread,
        NULL,
        4000
    );
}

void xml_access_module_shutdown(void)
{
}

/*!
** @}
*/

/*!
** @}
*/
