
/*****************************************************************************/
/*                              Legal                                        */
/*****************************************************************************/

/*
** Copyright 2015-2025, Lantronix, Inc. All Rights Reserved.
** By using this software, you are agreeing to the terms of the Software
** Development Kit (SDK) License Agreement included in the distribution package
** for this software (the License Agreement).
** Under the License Agreement, this software may be used solely to create
** custom applications for use on the Lantronix xPico Wi-Fi product.
** THIS SOFTWARE AND ANY ACCOMPANYING DOCUMENTATION IS PROVIDED "AS IS".
** LANTRONIX SPECIFICALLY DISCLAIMS ANY WARRANTIES, INCLUDING, BUT NOT LIMITED
** TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY, NON-INFRINGEMENT AND FITNESS
** FOR A PARTICULAR PURPOSE.
** LANTRONIX HAS NO OBLIGATION TO PROVIDE MAINTENANCE, SUPPORT, UPDATES,
** ENHANCEMENTS, OR MODIFICATIONS TO THIS SOFTWARE.
** IN NO EVENT SHALL LANTRONIX BE LIABLE TO ANY PARTY FOR DIRECT, INDIRECT,
** SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES, INCLUDING LOST PROFITS,
** ARISING OUT OF THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF
** LANTRONIX HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
*/

/*****************************************************************************/
/*                             Includes                                      */
/*****************************************************************************/

#include <stdio.h> /* MinGW plus Lantronix changes, delivered with SDK. */
#include <string.h> /* MinGW, delivered with SDK. */

#include "ltrx_network.h" /* Delivered with SDK. */
#include "ltrx_scan.h" /* Delivered with SDK. */
#include "ltrx_tlog.h" /* Delivered with SDK. */
#include "programmatic_scan_module_defs.h" /* Automatically generated by make. */

/*****************************************************************************/
/*                           Documentation                                   */
/*****************************************************************************/

/*!
** \addtogroup example
** @{
*/

/*!
** \defgroup programmatic_scan programmatic_scan
** @{
**
** The \b programmatic_scan module demonstrates how to run a wlan scan from
** your program and capture its output for program use.
**
** Build it from project "programmaticScanDemo".
*/

/*****************************************************************************/
/*                         Local Constants                                   */
/*****************************************************************************/

static const uint32_t s_delayTimeInMilliseconds = 20000;

/*****************************************************************************/
/*                               Code                                        */
/*****************************************************************************/

static void processScanResults(
    const char *ssid,
    const uint8_t *bssid,
    uint8_t channel,
    int rssi,
    unsigned char security_suite,
    unsigned char encr_flags
#ifdef USE_ENTERPRISE_AUTH
    ,
    bool use_enterprise_auth
#endif
)
{
    TLOG(TLOG_SEVERITY_LEVEL__DEBUG, "SSID: %s", ssid);
    TLOG(
        TLOG_SEVERITY_LEVEL__DEBUG,
        "  BSSID: %02X:%02X:%02X:%02X:%02X:%02X",
        bssid[0], bssid[1], bssid[2], bssid[3], bssid[4], bssid[5]
    );
    TLOG(TLOG_SEVERITY_LEVEL__DEBUG, "  Channel: %u", channel);
    TLOG(TLOG_SEVERITY_LEVEL__DEBUG, "  RSSI: %d dBm", rssi);
    TLOG(
        TLOG_SEVERITY_LEVEL__DEBUG,
        "  Security Suite: %s%s%s%s",
        ! (security_suite & (LTRX_SCAN_SECURITY_SUITE__WPA2|LTRX_SCAN_SECURITY_SUITE__WPA|LTRX_SCAN_SECURITY_SUITE__WEP)) ? "None" :
        (security_suite & LTRX_SCAN_SECURITY_SUITE__WPA2) ? "WPA2" :
        (security_suite & LTRX_SCAN_SECURITY_SUITE__WPA) ? "WPA" :
        (security_suite & LTRX_SCAN_SECURITY_SUITE__WEP) ? "WEP" :
        "",
        (security_suite & (LTRX_SCAN_SECURITY_SUITE__WPA2 | LTRX_SCAN_SECURITY_SUITE__WPA)) &&
        (encr_flags & LTRX_SCAN_ENCR_FLAGS__CCMP) ? "-CCMP" : "",
        (security_suite & (LTRX_SCAN_SECURITY_SUITE__WPA2 | LTRX_SCAN_SECURITY_SUITE__WPA)) &&
        (encr_flags & LTRX_SCAN_ENCR_FLAGS__TKIP) ? "-TKIP" : "",
#ifdef USE_ENTERPRISE_AUTH
        (security_suite & (LTRX_SCAN_SECURITY_SUITE__WPA2 | LTRX_SCAN_SECURITY_SUITE__WPA)) &&
        use_enterprise_auth ? "-EAP" : ""
#else
        ""
#endif
    );
}

static void programmaticScanThread(void *opaque)
{
    (void)opaque;
    while(true)
    {
	    while(ltrx_ip_address_state(NETS_WLAN_START) == 0)
	    {
		    ltrx_thread_sleep(1000); /* wlan0 not up yet. */
	    }
        if(ltrx_scan_begin(5000))
        {
            TLOG(TLOG_SEVERITY_LEVEL__DEBUG, "Starting scan.");
            ltrx_scan_collect(NULL, NULL);
            for(unsigned int i = 0; true; ++i)
            {
                struct ltrx_scan_result *lsr = ltrx_scan_get(i);
                if(lsr)
                {
                    processScanResults(
                        (const char *)lsr->ssid,
                        lsr->bssid,
                        lsr->channel,
                        lsr->rssi,
                        lsr->security_suite,
                        lsr->encr_flags
#ifdef USE_ENTERPRISE_AUTH
                        ,
                        lsr->use_enterprise_auth
#endif
                    );
                }
                else
                {
                    break;
                }
            }
            ltrx_scan_end();
            TLOG(TLOG_SEVERITY_LEVEL__DEBUG, "Scan completed.");
        }
        else
        {
            TLOG(TLOG_SEVERITY_LEVEL__DEBUG, "Device is busy servicing another scan request.");
		}
        ltrx_thread_sleep(s_delayTimeInMilliseconds);
    }
}

void programmatic_scan_module_registration(void)
{
    ltrx_module_register(&g_programmatic_scanModuleInfo);
}

void programmatic_scan_module_startup(void)
{
    ltrx_thread_create(
        "Programatic Scan",
        programmaticScanThread,
        NULL,
        4000
    );
}

void programmatic_scan_module_shutdown(void)
{
}

/*! @} End of Group. */
/*! @} End of Group. */
