
/*****************************************************************************/
/*                              Legal                                        */
/*****************************************************************************/

/*
** Copyright 2015-2025, Lantronix, Inc. All Rights Reserved.
** By using this software, you are agreeing to the terms of the Software
** Development Kit (SDK) License Agreement included in the distribution package
** for this software (the License Agreement).
** Under the License Agreement, this software may be used solely to create
** custom applications for use on the Lantronix xPico Wi-Fi product.
** THIS SOFTWARE AND ANY ACCOMPANYING DOCUMENTATION IS PROVIDED "AS IS".
** LANTRONIX SPECIFICALLY DISCLAIMS ANY WARRANTIES, INCLUDING, BUT NOT LIMITED
** TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY, NON-INFRINGEMENT AND FITNESS
** FOR A PARTICULAR PURPOSE.
** LANTRONIX HAS NO OBLIGATION TO PROVIDE MAINTENANCE, SUPPORT, UPDATES,
** ENHANCEMENTS, OR MODIFICATIONS TO THIS SOFTWARE.
** IN NO EVENT SHALL LANTRONIX BE LIABLE TO ANY PARTY FOR DIRECT, INDIRECT,
** SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES, INCLUDING LOST PROFITS,
** ARISING OUT OF THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF
** LANTRONIX HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
*/

/*****************************************************************************/
/*                           Documentation                                   */
/*****************************************************************************/

/*!
** \addtogroup example
** @{
*/

/*!
** \defgroup power_down power_down
** @{
**
** The \b power_down module demonstrates how an application program can place
** restrictions on when the device powers down.
**
** The application developer inserts code after "Insert your work here...".
** For example, the code might compose status and send it to a remote server.
** The work can take a variable length of time to complete.
** The processor will be kept powered up while the work is being performed.
** Then power down is allowed till
** \c s_timeBetweenWorkStartInMilliseconds has elapsed.
**
** Build it from project "powerDownDemo".
**
** To run it, go to Power Configuration, check the box in front of
** "Power Down Demo", enabling this application to hold the power on.
*/

/*****************************************************************************/
/*                             Includes                                      */
/*****************************************************************************/

#include "ltrx_network.h" /* Delivered with SDK. */
#include "ltrx_tlog.h" /* Delivered with SDK. */
#include "power_down_module_defs.h" /* Automatically generated by make. */

/*****************************************************************************/
/*                         Local Constants                                   */
/*****************************************************************************/

static const uint32_t s_timeBetweenWorkStartInMilliseconds = (
    2 * 60 * 1000 /* 2 minutes */
);

static const char s_applicationName[] = "Power Down Demo";

static const char s_powerLevelApplicationHelp[] = "is an SDK example.";

/*****************************************************************************/
/*                               Code                                        */
/*****************************************************************************/

static void powerDownThread(void *opaque)
{
    (void)opaque;
    while(true)
    {
	    while(ltrx_ip_address_state(NETS_WLAN_START) == 0)
	    {
		    ltrx_thread_sleep(100); /* wlan0 not up yet. */
	    }
        ltrx_power_level_application_request_full_power(s_applicationName);
        uint32_t timemark = ltrx_timemark();
        TLOG(TLOG_SEVERITY_LEVEL__DEBUG, "Starting work.");
        /* Insert your work here... */
        TLOG(TLOG_SEVERITY_LEVEL__DEBUG, "Work completed.");
        uint32_t elapsedMsec;
        if(
            (elapsedMsec = ltrx_elapsed_time_current_ms(timemark)) <
            s_timeBetweenWorkStartInMilliseconds
        )
        {
            uint32_t remainMsec = (
                s_timeBetweenWorkStartInMilliseconds - elapsedMsec
            );
            if(remainMsec > 1000)
            {
                uint32_t permitSeconds = (remainMsec + 500) / 1000;
                TLOG(
                    TLOG_SEVERITY_LEVEL__DEBUG,
                    "Permitting shutdown for %lu seconds.",
                    permitSeconds
                );
                ltrx_power_level_application_permit_shutdown(
                    s_applicationName, permitSeconds);
            }
            ltrx_thread_sleep(remainMsec);
        }
        ltrx_thread_yield();
    }
}

void power_down_module_registration(void)
{
    ltrx_module_register(&g_power_downModuleInfo);
    static const struct ltrx_power_level_application s_app =
    {
        .applicationName = s_applicationName,
        .helpHtml = s_powerLevelApplicationHelp
    };
    ltrx_power_level_application_register(&s_app);
}

void power_down_module_startup(void)
{
    ltrx_thread_create(
        "Power Down",
        powerDownThread,
        NULL,
        4000
    );
}

void power_down_module_shutdown(void)
{
}

/*!
** @}
*/

/*!
** @}
*/
