
/*****************************************************************************/
/*                              Legal                                        */
/*****************************************************************************/

/*
** Copyright 2015-2025, Lantronix, Inc. All Rights Reserved.
** By using this software, you are agreeing to the terms of the Software
** Development Kit (SDK) License Agreement included in the distribution package
** for this software (the License Agreement).
** Under the License Agreement, this software may be used solely to create
** custom applications for use on the Lantronix xPico Wi-Fi product.
** THIS SOFTWARE AND ANY ACCOMPANYING DOCUMENTATION IS PROVIDED "AS IS".
** LANTRONIX SPECIFICALLY DISCLAIMS ANY WARRANTIES, INCLUDING, BUT NOT LIMITED
** TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY, NON-INFRINGEMENT AND FITNESS
** FOR A PARTICULAR PURPOSE.
** LANTRONIX HAS NO OBLIGATION TO PROVIDE MAINTENANCE, SUPPORT, UPDATES,
** ENHANCEMENTS, OR MODIFICATIONS TO THIS SOFTWARE.
** IN NO EVENT SHALL LANTRONIX BE LIABLE TO ANY PARTY FOR DIRECT, INDIRECT,
** SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES, INCLUDING LOST PROFITS,
** ARISING OUT OF THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF
** LANTRONIX HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
*/

/*****************************************************************************/
/*                           Documentation                                   */
/*****************************************************************************/

/*!
** \addtogroup example
** @{
*/

/*!
** \defgroup line_to_mqtt line_to_mqtt
** @{
**
** The \b line_to_mqtt_example module implements a "Line Protocol" named "MQTT".
** It requires configuration setup in "MQTT".
** When this protocol is chosen by a Line, it buffers
** whatever it receives on the Line till it sees a newline,
** then sends the buffer contents as an MQTT publish.
**
** Build it from project "line_to_mqtt".
*/

/*****************************************************************************/
/*                             Includes                                      */
/*****************************************************************************/

#include <string.h>

#include "line_to_mqtt_example_module_defs.h" /* Automatically generated by make. */
#include "ltrx_line.h" /* Delivered with SDK. */
#include "ltrx_mqtt_client.h" /* Delivered with SDK. */
#include "ltrx_stream.h" /* Delivered with SDK. */
#include "ltrx_tlog.h" /* Delivered with SDK. */
#include "ltrx_webmgr.h" /* Delivered with SDK. */
#include "mqtt_module_libs.h" /* Delivered with SDK. */
#include "security_module_libs.h" /* Delivered with SDK. */

/*****************************************************************************/
/*                              Defines                                      */
/*****************************************************************************/

#define MQTT_CONNECTION_MAX_RETRIES 5

#define MQTT_CONNECTION_WAIT_MSEC 3000

#define MQTT_INPUT_BUFFER_SIZE 1000

#define MQTT_KEEPALIVE_SECONDS (4 * 60) /* 4 minutes */

#define MQTT_REPLY_BUFFER_SIZE 100

#define MQTT_TIMEOUT_MSEC 12000

#define TOPIC_LENGTH 128

/*****************************************************************************/
/*                             Structs                                       */
/*****************************************************************************/

struct thread_info
{
    uint32_t zeroBasedIndex;
    struct ltrx_thread *threadForLine;
    bool isRunning;
    struct ltrx_trigger eventTrigger;
    struct output_stream_to_uart ostu;
    struct input_stream_from_uart isfu;
    char inputBuffer[MQTT_INPUT_BUFFER_SIZE];
    char replyBuffer[MQTT_REPLY_BUFFER_SIZE];
    size_t charsInBuffer;
    struct ltrx_mqtt_client *lmc;
    struct vardef_values_mqtt config;
    struct statusdef_values_mqtt status;
    uint32_t conn_start_timemark;
    uint16_t send_message_id;
    char sawNewlineChar;
};

/*****************************************************************************/
/*                            Prototypes                                     */
/*****************************************************************************/

bool StartLineProtocol(uint16_t zeroBasedIndex);

void StopLineProtocol(uint16_t zeroBasedIndex);

/*****************************************************************************/
/*                         Local Constants                                   */
/*****************************************************************************/

static const struct ltrx_line_protocol s_lineProtocol =
{
    .protocolName = "MQTT",
    .helpHtml = "Buffer characters till newline; send as MQTT publish.",
    .startProtocol = StartLineProtocol,
    .stopProtocol = StopLineProtocol
};

/*****************************************************************************/
/*                         Local Variables                                   */
/*****************************************************************************/

static struct thread_info *s_threadInfo[MAX_LOGICAL_SERIAL_LINES];

/*****************************************************************************/
/*                          Global Variables                                 */
/*****************************************************************************/

const struct mqtt_external_functions *g_mqttExternalFunctionEntry_pointer = NULL;

const struct security_external_functions *g_securityExternalFunctionEntry_pointer = NULL;

/*****************************************************************************/
/*                               Code                                        */
/*****************************************************************************/

const char *LineName(unsigned int zeroBasedIndex)
{
    return ltrx_line_get_registered_name(zeroBasedIndex);
}

void line_to_mqtt_example_module_registration(void)
{
    ltrx_module_register(&g_line_to_mqtt_exampleModuleInfo);
    ltrx_line_register_protocol(&s_lineProtocol);
    ltrx_webmgr_init_js_register("/embedded/line_to_mqtt_example/http/web/mqtt_init.js");
}

void line_to_mqtt_example_module_startup(void)
{
    g_mqttExternalFunctionEntry_pointer = ltrx_module_functions_lookup("mqtt");
    if(! g_mqttExternalFunctionEntry_pointer)
    {
        TLOG(TLOG_SEVERITY_LEVEL__ERROR, "Failed lookup for module \"MQTT\".");
    }
    g_securityExternalFunctionEntry_pointer = ltrx_module_functions_lookup("Security");
    if(! g_securityExternalFunctionEntry_pointer)
    {
        TLOG(TLOG_SEVERITY_LEVEL__ERROR, "Failed lookup for module \"Security\".");
    }
}

void line_to_mqtt_example_module_shutdown(void)
{
}

static bool sendData(struct thread_info *ti)
{
    bool rc = false;
    if(ti->charsInBuffer)
    {
        struct ltrx_mqtt_message lwtmsg = {
            .qos = LTRX_MQTT_QOS1,
            .retained = 0,
            .dup = 0,
            .id = ti->send_message_id++,
            .payload = ti->inputBuffer,
            .payloadlen = ti->charsInBuffer,
        };
        rc = (
            ltrx_mqtt_client_publish(
                ti->lmc,
                ti->config.publish_topic,
                &lwtmsg
            ) == LTRX_MQTT_CLIENT_RC__SUCCESS
        );
    }
    return rc;
}

static void handleData(struct thread_info *ti)
{
    while(ltrx_input_stream_peek(&ti->isfu.inStream) >= 0)
    {
        char c = ltrx_input_stream_read(&ti->isfu.inStream);
        if(ti->sawNewlineChar)
        {
            if(
                ti->sawNewlineChar != c &&
                (c == '\r' || c == '\n' || c == 0)
            )
            {
                ti->sawNewlineChar = 0;
                return;
            }
            ti->sawNewlineChar = 0;
        }
        if(c == '\r' || c == '\n')
        {
            ti->sawNewlineChar = c;
            c = '\r';
        }
        if(c == '\r')
        {
            if(sendData(ti))
            {
                ++ti->status.publish_out;
            }
            else
            {
                ++ti->status.publish_fail;
            }
            ti->charsInBuffer = 0;
        }
        else if(ti->charsInBuffer < sizeof(ti->inputBuffer))
        {
            ti->inputBuffer[ti->charsInBuffer++] = c;
        }
    }
}

static bool mqtt_start(struct thread_info *ti)
{
    int rc;
    struct ltrx_mqtt_client_network_info lmcni = {
        .protocol = ti->config.protocol,
        .host = ti->config.broker_address,
        .url = ti->config._url.url,
        .host_port = ti->config.broker_port,
        .local_port = ti->config.local_port,
        .trusted_auth_array = NULL,
        .tls_name = ti->config._cred.credential_name
    };
    ltrx_mqtt_client_disconnect(ti->lmc);
    rc = ltrx_mqtt_client_set_network_info(ti->lmc, &lmcni);
    if(rc != 0)
    {
        TLOG(
            TLOG_SEVERITY_LEVEL__ERROR,
            "Failed setting MQTT network info: %d",
            rc
        );
        return false;
    }
    for(uint8_t i = 0; i < MQTT_CONNECTION_MAX_RETRIES; ++i)
    {
        struct ltrx_mqtt_client_connect_info lmcci = {
            .client_id = ti->config.device_id,
            .username = *ti->config.user_name ? ti->config.user_name : NULL,
            .password = *ti->config.auth_token ? ti->config.auth_token : NULL,
            .version = LTRX_MQTT_VERSION_3_1_1,
            .cleansession = true,
            .keepalive_interval_sec = MQTT_KEEPALIVE_SECONDS,
        };
        rc = ltrx_mqtt_client_set_connect_info(ti->lmc, &lmcci);
        if(rc != 0)
        {
            TLOG(
                TLOG_SEVERITY_LEVEL__ERROR,
                "Failed setting MQTT connection info: %d",
                rc
            );
            return false;
        }
        rc = ltrx_mqtt_client_connect(ti->lmc);
        if(rc == 0)
        {
            break;
        }
        ltrx_thread_sleep(MQTT_CONNECTION_WAIT_MSEC);
    }
    if(rc != 0)
    {
        TLOG(
            TLOG_SEVERITY_LEVEL__ERROR,
            "Failed to connect to MQTT broker."
        );
        return false;
    }
    TLOG(
        TLOG_SEVERITY_LEVEL__INFORMATIONAL,
        "Connected to MQTT broker."
    );
    return true;
}

static bool is_valid_account_info(
    const struct vardef_values_mqtt *config,
    const struct ltrx_write_user_message_info *lwumi
)
{
    if(! *config->broker_address)
    {
        ltrx_write_user_message(
            lwumi, LTRX_USER_MESSAGE_SEVERITY__WARNING,
            "Missing MQTT Broker Address."
        );
        return false;
    }
    if(! config->broker_port)
    {
        ltrx_write_user_message(
            lwumi, LTRX_USER_MESSAGE_SEVERITY__WARNING,
            "Missing MQTT Broker Port."
        );
        return false;
    }
    if(
        config->protocol == VARDEF_ENUM_MQTT_PROTOCOL__TLS
#ifdef VARDEF_ENUM_MQTT_PROTOCOL__WEBSOCKET_SECURE
        ||
        config->protocol == VARDEF_ENUM_MQTT_PROTOCOL__WEBSOCKET_SECURE
#endif
    )
    {
        if(! *config->_cred.credential_name)
        {
            ltrx_write_user_message(
                lwumi, LTRX_USER_MESSAGE_SEVERITY__WARNING,
                "TLS Credential name is required."
            );
            return false;
        }
        if(! ltrx_ssl_credential_is_present(config->_cred.credential_name))
        {
            ltrx_write_user_message(
                lwumi, LTRX_USER_MESSAGE_SEVERITY__WARNING,
                "Not finding TLS Credential \"{}\".",
                config->_cred.credential_name
            );
            return false;
        }
    }
    if(! *config->device_id)
    {
        ltrx_write_user_message(
            lwumi, LTRX_USER_MESSAGE_SEVERITY__WARNING,
            "Missing MQTT Device ID."
        );
        return false;
    }
    unsigned int subscribeTopics = 0;
    for(unsigned int i = 0; i < MAX_MQTT_SUBSCRIBE_TOPICS; ++i)
    {
        if(*config->subscribe[i].topic)
        {
            ++subscribeTopics;
        }
    }
    if(! *config->publish_topic && ! subscribeTopics)
    {
        ltrx_write_user_message(
            lwumi, LTRX_USER_MESSAGE_SEVERITY__WARNING,
            "Missing MQTT publish / Subscribe Topic."
        );
        return false;
    }
    return true;
}

static void mqtt_subscription_handler(
    struct ltrx_mqtt_message_data *lmmd,
    void *opaque
)
{
    uint32_t zeroBasedIndex = (uint32_t)opaque;
    bool r = false;
    struct thread_info *ti = s_threadInfo[zeroBasedIndex];
    for(unsigned int i = 0; i < MAX_MQTT_SUBSCRIBE_TOPICS; ++i)
    {
        const char *name = ti->config.subscribe[i].topic;
        if(*name)
        {
            if(0 == strncmp(
                    name,
                    lmmd->topic->len_string.data,
                    strlen(name)
                )
            )
            {
                r = ltrx_output_stream_write_line(&ti->ostu.outStream, name);
                if(r)
                {
                    r = ltrx_output_stream_write_line(&ti->ostu.outStream, "&");
                }
                if(r)
                {
                    r = ltrx_output_stream_write_binary(
                        &ti->ostu.outStream,
                        lmmd->message->payload,
                        lmmd->message->payloadlen
                    );
                }
                if(r)
                {
                    r = ltrx_output_stream_write_without_ending_line(
                        &ti->ostu.outStream, "\n"
                    );
                }
            }
        }
    }
    if(r)
    {
        ++ti->status.subscribe_in;
    }
    else
    {
        ++ti->status.subscribe_discard;
    }
}

static bool mqtt_stop(struct thread_info *ti)
{
    int rc = 0;
    if(
        ti->status.state == STATUSDEF_ENUM_MQTT_STATE__UP &&
        (rc = ltrx_mqtt_client_disconnect(ti->lmc)) != 0
    )
    {
        TLOG(
            TLOG_SEVERITY_LEVEL__ERROR,
            "Failed to disconnect from MQTT broker: %d",
            rc
        );
    }
    TLOG(
        TLOG_SEVERITY_LEVEL__INFORMATIONAL,
        "Disconnected from MQTT broker."
    );
    ti->status.state = STATUSDEF_ENUM_MQTT_STATE__DOWN;
    return rc == 0;
}

static bool iot_disconnect(struct thread_info *ti)
{
    return mqtt_stop(ti);
}

static bool iot_connect(struct thread_info *ti)
{
    if(! is_valid_account_info(&ti->config, NULL))
    {
        return false;
    }
    ti->status.state = STATUSDEF_ENUM_MQTT_STATE__CONNECTING;
    if(! mqtt_start(ti))
    {
        return false;
    }
    ti->status.state = STATUSDEF_ENUM_MQTT_STATE__UP;
    for(unsigned int i = 0; i < MAX_MQTT_SUBSCRIBE_TOPICS; ++i)
    {
        const char *name = ti->config.subscribe[i].topic;
        if(*name)
        {
            int rc = ltrx_mqtt_client_subscribe(
                    ti->lmc,
                    name,
                    LTRX_MQTT_QOS1,
                    (void *)ti->zeroBasedIndex,
                    mqtt_subscription_handler
            );
            if(LTRX_MQTT_CLIENT_RC__SUCCESS != rc)
            {
                TLOG(
                    TLOG_SEVERITY_LEVEL__ERROR,
                    "Failure subscribing to MQTT broker,rc = %d, Topic: %s",
                    rc, name
                );
                ltrx_mqtt_client_unsubscribe(ti->lmc, name);
                iot_disconnect(ti);
            }
            else if(LTRX_MQTT_CLIENT_RC__SUCCESS == rc)
            {
                TLOG(
                    TLOG_SEVERITY_LEVEL__INFORMATIONAL,
                    "Subscribed to MQTT broker, Topic: %s.",
                    name
                );
            }
        }
    }
    ti->conn_start_timemark = ltrx_timemark();
    return true;
}

static void lineLoop(struct thread_info *ti)
{
    ltrx_input_stream_init_from_uart(&ti->isfu, ti->zeroBasedIndex);
    ltrx_output_stream_init_to_uart(&ti->ostu, ti->zeroBasedIndex);
    ltrx_output_stream_write_line(&ti->ostu.outStream, "");
    ltrx_output_stream_write_line(
        &ti->ostu.outStream, s_lineProtocol.protocolName
    );
    ltrx_output_stream_write_line(&ti->ostu.outStream, "Ready");
    while(ti->isRunning)
    {
        uint32_t timeoutMilliseconds = 1000;
        switch(ti->status.state)
        {
        case STATUSDEF_ENUM_MQTT_STATE__DOWN:
        case STATUSDEF_ENUM_MQTT_STATE__CONNECTING:
            if(iot_connect(ti))
            {
                ++ti->status.connections;
            }
            else
            {
                timeoutMilliseconds = 10000;
                ++ti->status.connect_fail;
            }
            break;
        case STATUSDEF_ENUM_MQTT_STATE__UP:
            if(! ltrx_mqtt_client_is_connected(ti->lmc))
            {
                iot_disconnect(ti);
                continue;
            }
            break;
        default:
            break;
        }
        if(
            ltrx_line_read_bytes_available(
                ti->zeroBasedIndex, &ti->eventTrigger
            )
        )
        {
            handleData(ti);
        }
        LTRX_TRIGGER_WAIT(&ti->eventTrigger, timeoutMilliseconds);
        ltrx_trigger_clear(&ti->eventTrigger);
    }
    iot_disconnect(ti);
}

static void lineThread(void *opaque)
{
    line_to_mqtt_example_module_startup();
    struct thread_info *ti = opaque;
    ltrx_cfgvar_get_data(
        ti->zeroBasedIndex,
        &g_line_to_mqtt_exampleModuleInfo.config_table[VARDEF_ELEMENT_GROUP_MQTT],
        &ti->config
    );
    uint16_t line = ti->zeroBasedIndex + 1;
    bool loggedStartMessage = false;
    ti->isRunning = true;
    if(! ltrx_trigger_create(&ti->eventTrigger, s_lineProtocol.protocolName))
    {
        return;
    }
    ti->lmc = ltrx_mqtt_client_create(
        MQTT_TIMEOUT_MSEC,
        sizeof(ti->inputBuffer) + 50,
        sizeof(ti->replyBuffer)
    );
    if(! ti->lmc)
    {
        TLOG(
            TLOG_SEVERITY_LEVEL__ERROR,
            "Failed to create MQTT Client"
        );
        ltrx_trigger_destroy(&ti->eventTrigger);
        return;
    }
    ltrx_mqtt_client_set_log_verbosity(ti->lmc, ti->config.diagnostic_log);
    while(
        ti->isRunning &&
        ! ltrx_line_open(ti->zeroBasedIndex, 1000)
    );
    if(ti->isRunning)
    {
        TLOG(
            TLOG_SEVERITY_LEVEL__INFORMATIONAL,
            "%s started on Line %u",
            s_lineProtocol.protocolName,
            line
        );
        loggedStartMessage = true;
        ltrx_line_set_dtr(ti->zeroBasedIndex, true);
        lineLoop(ti);
    }
    if(loggedStartMessage)
    {
        TLOG(
            TLOG_SEVERITY_LEVEL__INFORMATIONAL,
            "%s stopped on Line %u",
            s_lineProtocol.protocolName,
            line
        );
    }
    ltrx_line_close(ti->zeroBasedIndex);
    ti->status.state = STATUSDEF_ENUM_MQTT_STATE__DOWN;
    ltrx_mqtt_client_destroy(ti->lmc);
    ltrx_trigger_destroy(&ti->eventTrigger);
    ti->threadForLine = NULL;
}

bool StartLineProtocol(uint16_t zeroBasedIndex)
{
    struct thread_info *ti;
    if(zeroBasedIndex >= MAX_LOGICAL_SERIAL_LINES)
    {
        return false;
    }
    uint16_t line = zeroBasedIndex + 1;
    ltrx_preemption_block();
    ti = s_threadInfo[zeroBasedIndex];
    if(! ti)
    {
        ti = MALLOC(sizeof(*ti));
    }
    if(ti)
    {
        memset(ti, 0, sizeof(*ti));
        ti->zeroBasedIndex = zeroBasedIndex;
        s_threadInfo[zeroBasedIndex] = ti;
    }
    ltrx_preemption_unblock();
    if(! ti)
    {
        return false;
    }
    if(ti->threadForLine)
    {
        TLOG(
            TLOG_SEVERITY_LEVEL__ERROR,
            "%s thread already running for Line %u",
            s_lineProtocol.protocolName,
            line
        );
        return false;
    }
    ti->threadForLine = ltrx_thread_create(
        s_lineProtocol.protocolName,
        lineThread,
        ti,
        5000
    );
    if(! ti->threadForLine)
    {
        TLOG(
            TLOG_SEVERITY_LEVEL__ERROR,
            "Failed to create %s thread for Line %u",
            s_lineProtocol.protocolName,
            line
        );
        return false;
    }
    return true;
}

void StopLineProtocol(uint16_t zeroBasedIndex)
{
    if(zeroBasedIndex < MAX_LOGICAL_SERIAL_LINES)
    {
        struct thread_info *ti = s_threadInfo[zeroBasedIndex];
        if(ti)
        {
            bool wasRunning = false;
            ltrx_preemption_block();
            if(ti->isRunning)
            {
                wasRunning = true;
                ti->isRunning = false;
                ltrx_trigger_signal(&ti->eventTrigger);
            }
            ltrx_preemption_unblock();
            if(wasRunning)
            {
                uint32_t tm = ltrx_timemark();
                while(
                    ti->threadForLine &&
                    ti->threadForLine != ltrx_thread_id() &&
                    ltrx_elapsed_time_current_ms(tm) < 2000
                )
                {
                    if(ltrx_elapsed_time_current_ms(tm) >= 500)
                    {
                        ltrx_line_purge(zeroBasedIndex);
                    }
                    ltrx_thread_sleep(100);
                }
            }
        }
    }
}

bool LineToMqttTestSet(
    uint32_t zeroBasedIndex,
    struct vardef_values_mqtt *config,
    bool set,
    const struct ltrx_write_user_message_info *lwumi
)
{
    (void) lwumi;
    if(set)
    {
        struct thread_info *ti = s_threadInfo[zeroBasedIndex];
        if(ti)
        {
            if(
                ti->isRunning &&
                ti->status.state != STATUSDEF_ENUM_MQTT_STATE__DOWN &&
                memcmp(config, &ti->config, sizeof(*config)) != 0
            )
            {
                if(ti->status.state == STATUSDEF_ENUM_MQTT_STATE__UP)
                {
                    if(
                        ! ltrx_write_user_message(
                            lwumi, LTRX_USER_MESSAGE_SEVERITY__CONFIRM,
                            "This will kill an active connection. Continue?"
                        )
                    )
                    {
                        return false;
                    }
                }
                else
                {
                    if(
                        ! ltrx_write_user_message(
                            lwumi, LTRX_USER_MESSAGE_SEVERITY__CONFIRM,
                            "This will kill a connection in progress. Continue?"
                        )
                    )
                    {
                        return false;
                    }
                }
                ti->isRunning = false;
                ltrx_line_config_request_protocol_change(zeroBasedIndex);
                ltrx_trigger_signal(&ti->eventTrigger);
            }
            ti->config = *config;
        }
    }
    return true;
}

void LineToMqttWarnings(
    uint32_t zeroBasedIndex,
    const struct ltrx_write_user_message_info *lwumi
)
{
    if(
        ! ltrx_config_value_of_is_(
            zeroBasedIndex,
            &g_mainModuleInfo.config_table[
                VARDEF_ELEMENT_ITEM_LINE__DRIVER__DEVICE_PROTOCOL
            ],
            s_lineProtocol.protocolName
        )
    )
    {
        ltrx_write_user_message(
            lwumi,
            LTRX_USER_MESSAGE_SEVERITY__WARNING,
            "Line {} protocol is not \"{}\".",
            ltrx_line_get_registered_name(zeroBasedIndex),
            s_lineProtocol.protocolName
        );
        return;
    }
    if(! ltrx_line_warning_if_shutdown(zeroBasedIndex, lwumi))
    {
        /* Warning has been given by above routine. */
        return;
    }
    struct vardef_values_mqtt config;
    ltrx_cfgvar_get_data(
        zeroBasedIndex,
        &g_line_to_mqtt_exampleModuleInfo.config_table[VARDEF_ELEMENT_GROUP_MQTT],
        &config
    );
    is_valid_account_info(&config, lwumi);
}

bool MQTTSubscribeSummary(
    uint32_t zeroBasedIndex,
    struct output_stream *out
)
{
    struct vardef_values_mqtt_subscribe vvms;
    if(
        ! ltrx_cfgvar_get_data(
            zeroBasedIndex,
            &g_line_to_mqtt_exampleModuleInfo.config_table[
                VARDEF_ELEMENT_GROUP_MQTT_SUBSCRIBE
            ],
            &vvms
        )
    )
    {
        return false;
    }
    if(*vvms.topic)
    {
        ltrx_output_stream_write_without_ending_line(out, vvms.topic);
    }
    else
    {
        return false;
    }
    return true;
}

void LineToMqttGetStatus(
    uint32_t zeroBasedIndex,
    struct statusdef_values_mqtt *status,
    const struct ltrx_write_user_message_info *lwumi
)
{
    (void)lwumi;
    memset(status, 0, sizeof(*status));
    struct thread_info *ti = s_threadInfo[zeroBasedIndex];
    if(ti)
    {
        *status = ti->status;
        if(status->state == STATUSDEF_ENUM_MQTT_STATE__UP)
        {
            status->_isup.uptime = (
                ltrx_elapsed_time_current_ms(ti->conn_start_timemark) /
                1000
            );
        }
    }
}

/*!
** @}
*/

/*!
** @}
*/
