
/*****************************************************************************/
/*                              Legal                                        */
/*****************************************************************************/

/*
** Copyright ©2020-2025, Lantronix, Inc. All Rights Reserved.
** By using this software, you are agreeing to the terms of the Software
** Development Kit (SDK) License Agreement included in the distribution package
** for this software (the “License Agreement”).
** Under the License Agreement, this software may be used solely to create
** custom applications for use on the Lantronix xPico Wi-Fi product.
** THIS SOFTWARE AND ANY ACCOMPANYING DOCUMENTATION IS PROVIDED "AS IS".
** LANTRONIX SPECIFICALLY DISCLAIMS ANY WARRANTIES, INCLUDING, BUT NOT LIMITED
** TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY, NON-INFRINGEMENT AND FITNESS
** FOR A PARTICULAR PURPOSE.
** LANTRONIX HAS NO OBLIGATION TO PROVIDE MAINTENANCE, SUPPORT, UPDATES,
** ENHANCEMENTS, OR MODIFICATIONS TO THIS SOFTWARE.
** IN NO EVENT SHALL LANTRONIX BE LIABLE TO ANY PARTY FOR DIRECT, INDIRECT,
** SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES, INCLUDING LOST PROFITS,
** ARISING OUT OF THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF
** LANTRONIX HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
*/

/****************************************************************************/
/*                                 Includes                                 */
/****************************************************************************/

#include <inttypes.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include "ble_test_server_module_defs.h"
#include "bts_gatt.h"
#include "bts_utils.h"
#include "ltrx_cfgvar.h"
#include "ltrx_definitions.h"
#include "ltrx_bluetooth.h"
#include "ltrx_gatt.h"
#include "ltrx_stream.h"
#include "ltrx_tlog.h"


/****************************************************************************/
/*                           Function Definitions                           */
/****************************************************************************/

bool BLETestServerStart(
    uint32_t zeroBasedIndex,
    const struct ltrx_write_user_message_info *lwumi
)
{
    (void)zeroBasedIndex;
    (void)lwumi;
    bool rc = bts_start();
    if (! rc)
    {
        ltrx_write_user_message(
            lwumi,
            LTRX_USER_MESSAGE_SEVERITY__ERROR,
            "Failed to start BLE Test Server."
        );
    }
    return rc;
}

bool BLETestServerStop(
    uint32_t zeroBasedIndex,
    const struct ltrx_write_user_message_info *lwumi
)
{
    (void)zeroBasedIndex;
    (void)lwumi;
    bts_stop();
    return true;
}

bool BLETestServerClearScreen(
    uint32_t zeroBasedIndex,
    const struct ltrx_write_user_message_info *lwumi
)
{
    (void)zeroBasedIndex;
    if (lwumi->env == CFGVAR_TESTSET_ENVIRONMENT__CLI)
    {
        struct output_stream *os = ltrx_cli_get_output_stream(
            lwumi->optCliThreadInfo
        );
        ltrx_output_stream_write_without_ending_line(
            os, "\x1B[2J\x1B[;H"
        );
    }
    return true;
}

bool BLETestServerSetMode(
    uint32_t zeroBasedIndex,
    const char *value,
    const struct ltrx_write_user_message_info *lwumi
)
{
    (void)zeroBasedIndex;
    if (strcmp(value, "write back") == 0)
    {
        bts_mode_set(BTS_MODE__WRITEBACK);
        return true;
    }
    else if (strcmp(value, "data logging") == 0)
    {
        bts_mode_set(BTS_MODE__DATALOGGING);
        return true;
    }
    ltrx_write_user_message(
        lwumi,
        LTRX_USER_MESSAGE_SEVERITY__ERROR,
        "Must specify \"write back\" or \"data logging\"."
    );
    return false;
}

void BLETestServerStatus(
    uint32_t zeroBasedIndex,
    struct statusdef_values_ble_test_server *status,
    const struct ltrx_write_user_message_info *lwumi
)
{
    (void)zeroBasedIndex;
    (void)lwumi;

    memset(status, 0, sizeof(*status));
    status->state = bts_is_running();
    if (status->state)
    {
        status->_runningstatedata.mode = bts_mode_get();
        {
            struct ltrx_uuid lu = {
                .length = UUID_LENGTH_128,
                .uu = { .id128 = { UUID__TEST_SERVICE } },
            };
            bt_uuid_unparse(
                &lu,
                status->_runningstatedata.service_uuid,
                sizeof(status->_runningstatedata.service_uuid)
            );
        }
        {
            struct ltrx_uuid lu = {
                .length = UUID_LENGTH_128,
                .uu = { .id128 = { UUID__TEST_CHARACTERISTIC_READ_WRITE } },
            };
            bt_uuid_unparse(
                &lu,
                status->_runningstatedata.write_characteristic_uuid,
                sizeof(status->_runningstatedata.write_characteristic_uuid)
            );
        }
        {
            struct ltrx_uuid lu = {
                .length = UUID_LENGTH_128,
                .uu = { .id128 = { UUID__TEST_CHARACTERISTIC_READ_WRITE_NR } },
            };
            bt_uuid_unparse(
                &lu,
                status->_runningstatedata.write_no_response_characteristic_uuid,
                sizeof(status->_runningstatedata.write_no_response_characteristic_uuid)
            );
        }
        {
            struct ltrx_uuid lu = {
                .length = UUID_LENGTH_128,
                .uu = { .id128 = { UUID__TEST_CHARACTERISTIC_STATUS } },
            };
            bt_uuid_unparse(
                &lu,
                status->_runningstatedata.status_notify_indicate_characteristic_uuid,
                sizeof(status->_runningstatedata.status_notify_indicate_characteristic_uuid)
            );
        }

        status->_runningstatedata.client_connected = bts_is_client_connected();
        if (status->_runningstatedata.client_connected)
        {
            struct bts_stats bss;
            bts_get_stats(&bss);
            status->_runningstatedata._connection.mtu = bss.bs_mtu;
            status->_runningstatedata._connection.connection_id = bss.bs_connection_id;
            status->_runningstatedata._connection.bytes_read = bss.bs_nread;
            status->_runningstatedata._connection.bytes_written = bss.bs_nwritten;
            status->_runningstatedata._connection.queued_writes = bss.bs_queued_writes;
            status->_runningstatedata._connection.queued_write_bytes = bss.bs_queued_write_bytes;
        }
    }

}

