
/*
 * Copyright 2017-2025, Lantronix, Inc. All Rights Reserved.
 *
 * This is UNPUBLISHED PROPRIETARY SOURCE CODE of Lantronix, Inc.;
 * the contents of this file may not be disclosed to third parties, copied
 * or duplicated in any form, in whole or in part, without the prior
 * written permission of Lantronix, Inc.
 */

/****************************************************************************/
/*                                 Includes                                 */
/****************************************************************************/

#include <stdbool.h>
#include <stdint.h>
#include <string.h>

#include "bluetooth_module_libs.h"
#include "bts_gatt.h"
#include "gatt_definitions.h"
#include "ltrx_bluetooth.h"
#include "ltrx_cfgvar.h"
#include "ltrx_gatt.h"
#include "ltrx_gatt_server.h"
#include "ltrx_stream.h"
#include "ltrx_time.h"
#include "ltrx_tlog.h"
#include "uuid.h"

#pragma GCC diagnostic ignored "-Wunused-function"
#pragma GCC diagnostic ignored "-Wunused-parameter"
#pragma GCC diagnostic ignored "-Wunused-variable"


/****************************************************************************/
/*                                 Defines                                  */
/****************************************************************************/

#define BTS_SERVICE_NAME            "BLE Test Server"

#define BTS_MTU_SIZE_MAX            512

#define BTS_BUFFER_SIZE             512

#define BTS_PREPARED_WRITE_QUE_SIZE 4


/****************************************************************************/
/*                             Data Structures                              */
/****************************************************************************/

struct bts_prepared_write
{
    uint16_t bpw_handle;
    uint16_t bpw_length;
    uint16_t bpw_offset;
    uint8_t *bpw_data;
};

struct bts_connection
{
    uint16_t bc_connection_id;
    uint16_t bc_mtu_size;
    uint16_t bc_cccd;

    struct bts_stats bc_stats;

    uint8_t *bc_buffer1;
    uint16_t bc_buffer1_size;

    uint8_t *bc_buffer2;
    uint16_t bc_buffer2_size;

    uint8_t *bc_read_buffer;
    uint16_t bc_read_buffer_size;
    uint16_t bc_read_index;

    uint8_t *bc_readnr_buffer;
    uint16_t bc_readnr_buffer_size;
    uint16_t bc_readnr_index;

    uint8_t *bc_write_buffer;
    uint16_t bc_write_buffer_size;
    uint16_t bc_write_index;

    uint8_t *bc_writenr_buffer;
    uint16_t bc_writenr_buffer_size;
    uint16_t bc_writenr_index;

    uint8_t bc_prepared_write_count;
    struct bts_prepared_write bc_prepared_writes[BTS_PREPARED_WRITE_QUE_SIZE];
};

enum gatt_handle_test_server
{
    GH_BTS__SERVICE = LTRX_GATT_HANDLE_BASE,
    GH_BTS__CHARACTERISTIC_TEST_READ_WRITE,
    GH_BTS__CHARACTERISTIC_TEST_READ_WRITE_VALUE,
    GH_BTS__CHARACTERISTIC_TEST_READ_WRITE_NR,
    GH_BTS__CHARACTERISTIC_TEST_READ_WRITE_NR_VALUE,
    GH_BTS__CHARACTERISTIC_STATUS_NOTIFY,
    GH_BTS__CHARACTERISTIC_STATUS_NOTIFY_VALUE,
    GH_BTS__CHARACTERISTIC_CONFIG_DESCRIPTOR,
};


/****************************************************************************/
/*                                  Locals                                  */
/****************************************************************************/

#pragma GCC diagnostic push
#pragma GCC diagnostic ignored "-Wdiscarded-qualifiers"

static const struct ltrx_gatt_server_descriptor s_cccd =
{
    .uuid = { 2, { GATT_UUID_CHAR_CLIENT_CONFIG } },
    .handle = GH_BTS__CHARACTERISTIC_CONFIG_DESCRIPTOR,
    .permissions = ( 0
        | GATT_DB_PERMISSION__READABLE
        | GATT_DB_PERMISSION__WRITE_REQ
    ),
};

static const struct ltrx_gatt_server_characteristic s_characteristic[] =
{
    {
        .uuid = { 16, { .id128 = { UUID__TEST_CHARACTERISTIC_READ_WRITE } } },
        .handle = GH_BTS__CHARACTERISTIC_TEST_READ_WRITE,
        .value_handle = GH_BTS__CHARACTERISTIC_TEST_READ_WRITE_VALUE,
        .properties = ( 0
            | GATT_DB_PROPERTY__READ
            | GATT_DB_PROPERTY__WRITE
        ),
        .permissions = ( 0
            | GATT_DB_PERMISSION__READABLE
            | GATT_DB_PERMISSION__WRITE_REQ
            | GATT_DB_PERMISSION__RELIABLE_WRITE
            | GATT_DB_PERMISSION__VARIABLE_LENGTH
        ),
        .descriptor_count = 0,
        .descriptor_array = NULL,
    },
    {
        .uuid = { 16, { .id128 = { UUID__TEST_CHARACTERISTIC_READ_WRITE_NR } } },
        .handle = GH_BTS__CHARACTERISTIC_TEST_READ_WRITE_NR,
        .value_handle = GH_BTS__CHARACTERISTIC_TEST_READ_WRITE_NR_VALUE,
        .properties = ( 0
            | GATT_DB_PROPERTY__READ
            | GATT_DB_PROPERTY__WRITE_NO_RESPONSE
        ),
        .permissions = ( 0
            | GATT_DB_PERMISSION__READABLE
            | GATT_DB_PERMISSION__WRITE_CMD
            | GATT_DB_PERMISSION__RELIABLE_WRITE
            | GATT_DB_PERMISSION__VARIABLE_LENGTH
        ),
        .descriptor_count = 0,
        .descriptor_array = NULL,
    },
    {
        .uuid = { 16, { .id128 = { UUID__TEST_CHARACTERISTIC_STATUS } } },
        .handle = GH_BTS__CHARACTERISTIC_STATUS_NOTIFY,
        .value_handle = GH_BTS__CHARACTERISTIC_STATUS_NOTIFY_VALUE,
        .properties = ( 0
            | GATT_DB_PROPERTY__READ
            | GATT_DB_PROPERTY__NOTIFY
            | GATT_DB_PROPERTY__INDICATE
        ),
        .permissions = ( 0
            | GATT_DB_PERMISSION__READABLE
        ),
        .descriptor_count = 1,
        .descriptor_array = &s_cccd,
    },
};

static const struct ltrx_gatt_server_service s_service =
{
    .uuid = { 16, { .id128 = { UUID__TEST_SERVICE } } },
    .handle = GH_BTS__SERVICE,
    .advertise = true,
    .characteristic_count = ARRAY_SIZE(s_characteristic),
    .characteristic_array = s_characteristic,
};

#pragma GCC diagnostic pop


static struct bts_connection *s_connection;

static uint8_t s_mode;

static bool s_registered;

static struct ltrx_thread *s_thread;

static bool s_run_thread;

static struct ltrx_trigger s_trigger;

static bool s_respond;


/****************************************************************************/
/*                           Function Definitions                           */
/****************************************************************************/

static const char *bts_str_mode(uint8_t mode)
{
    switch (mode)
    {
    case BTS_MODE__WRITEBACK: return "write back";
    case BTS_MODE__DATALOGGING: return "data logging";
    default: return "<ERROR>";
    }
}

static void bts_mode_configure(uint8_t mode, struct bts_connection *bc)
{
    ltrx_preemption_block();
    memset(bc->bc_buffer1, 0, bc->bc_buffer1_size);
    memset(bc->bc_buffer2, 0, bc->bc_buffer2_size);
    if (mode == BTS_MODE__WRITEBACK)
    {
        bc->bc_write_buffer = bc->bc_buffer1;
        bc->bc_write_buffer_size = bc->bc_buffer1_size;
        bc->bc_write_index = 0;

        bc->bc_read_buffer = bc->bc_buffer1;
        bc->bc_read_buffer_size = bc->bc_buffer1_size;
        bc->bc_read_index = 0;

        bc->bc_writenr_buffer = bc->bc_buffer2;
        bc->bc_writenr_buffer_size = bc->bc_buffer2_size;
        bc->bc_writenr_index = 0;

        bc->bc_readnr_buffer = bc->bc_buffer2;
        bc->bc_readnr_buffer_size = bc->bc_buffer2_size;
        bc->bc_readnr_index = 0;
    }
    else if (mode == BTS_MODE__DATALOGGING)
    {
        bc->bc_write_buffer = NULL;
        bc->bc_write_buffer_size = 0;
        bc->bc_write_index = 0;

        bc->bc_read_buffer = bc->bc_buffer1;
        bc->bc_read_buffer_size = bc->bc_buffer1_size;
        bc->bc_read_index = 0;
        for (uint32_t i = 0; i < bc->bc_read_buffer_size; ++i)
        {
            s_connection->bc_read_buffer[i] = i & 0xFF;
        }
    }
    ltrx_preemption_unblock();
    TLOG(
        TLOG_SEVERITY_LEVEL__INFORMATIONAL, "Mode is now %s.", bts_str_mode(mode)
    );
}

static void bts_connection(uint16_t connection_id, bool connected)
{
    ltrx_preemption_block();
    if (connected)
    {
        s_connection = MALLOC_ZEROED(sizeof(*s_connection));
        if (! s_connection)
        {
            ltrx_preemption_unblock();
            return;
        }
        s_connection->bc_mtu_size = GATT_MTU_SIZE_DEFAULT;
        s_connection->bc_connection_id = connection_id;

        s_connection->bc_buffer1_size = BTS_BUFFER_SIZE;
        s_connection->bc_buffer1 = MALLOC(s_connection->bc_buffer1_size);

        s_connection->bc_buffer2_size = BTS_BUFFER_SIZE;
        s_connection->bc_buffer2 = MALLOC(s_connection->bc_buffer2_size);

        if (! s_connection->bc_buffer1 || ! s_connection->bc_buffer2)
        {
            ltrx_free(s_connection);
            s_connection = NULL;
            ltrx_preemption_unblock();
            return;
        }
        bts_mode_configure(s_mode, s_connection);
    }
    else
    {
        ltrx_free(s_connection->bc_buffer1);
        ltrx_free(s_connection->bc_buffer2);
        ltrx_free(s_connection);
        s_connection = NULL;
    }
    ltrx_preemption_unblock();
    TLOG(
        TLOG_SEVERITY_LEVEL__INFORMATIONAL,
        "Client %sconnected with id %u.",
        connected ? "" : "dis", connection_id
    );
}

static const char *bts_str_operation(uint8_t o)
{
    switch (o)
    {
    default:
    case GATT_OPERATION__NONE: return "none";
    case GATT_OPERATION__DISCOVERY: return "discovery";
    case GATT_OPERATION__READ: return "read";
    case GATT_OPERATION__WRITE: return "write";
    case GATT_OPERATION__WRITE_EXECUTE: return "write execute";
    case GATT_OPERATION__CONFIGURATION: return "configuration";
    case GATT_OPERATION__NOTIFICATION: return "notification";
    case GATT_OPERATION__INDICATION: return "indication";
    }
}

static int bts_attribute_log(
    uint8_t operation, uint8_t *data, uint16_t length, uint16_t offset
)
{
    uint32_t write_offset = offset;
    TLOG(
        TLOG_SEVERITY_LEVEL__INFORMATIONAL,
        "%u %s bytes @ offset %u:",
        length, bts_str_operation(operation), offset
    );
    TLOG_HEXDUMP(
        TLOG_SEVERITY_LEVEL__INFORMATIONAL,
        data,
        length,
        (void *)write_offset
    );
    return LTRX_GATT_STATUS__SUCCESS;
}

static int bts_attribute_read(
    uint16_t handle, uint16_t offset, uint8_t *datap, uint16_t *lengthp
)
{
    TLOG(
        TLOG_SEVERITY_LEVEL__DEBUG,
        "Read hnd: 0x%X  off: %u  len: %u",
        handle, offset, *lengthp
    );
    switch (handle)
    {
    case GH_BTS__CHARACTERISTIC_TEST_READ_WRITE_VALUE:
        if (s_mode == BTS_MODE__WRITEBACK)
        {
            if (offset > s_connection->bc_read_index)
            {
                return LTRX_GATT_STATUS__INVALID_OFFSET;
            }
            uint8_t *start = &s_connection->bc_read_buffer[offset];
            uint8_t *end = &s_connection->bc_read_buffer[s_connection->bc_read_index];
            if (start < end)
            {
                *lengthp = MINIMUM(end - start, *lengthp);
                memcpy(datap, start, *lengthp);
                s_connection->bc_stats.bs_nread += *lengthp;
            }
            else
            {
                *lengthp = 0;
            }
        }
        else if (s_mode == BTS_MODE__DATALOGGING)
        {
            if (s_connection->bc_read_buffer_size > offset)
            {
                *lengthp = MINIMUM(
                    s_connection->bc_read_buffer_size - offset, *lengthp
                );
                memcpy(datap, &s_connection->bc_read_buffer[offset], *lengthp);
                s_connection->bc_stats.bs_nwritten += *lengthp;
            }
            else
            {
                *lengthp = 0;
            }
            bts_attribute_log(GATT_OPERATION__READ, datap, *lengthp, offset);
        }
        break;

    case GH_BTS__CHARACTERISTIC_TEST_READ_WRITE_NR_VALUE:
        if (s_mode == BTS_MODE__WRITEBACK)
        {
            if (offset > s_connection->bc_readnr_index)
            {
                return LTRX_GATT_STATUS__INVALID_OFFSET;
            }
            uint8_t *start = &s_connection->bc_readnr_buffer[offset];
            uint8_t *end = &s_connection->bc_readnr_buffer[s_connection->bc_readnr_index];
            if (start < end)
            {
                *lengthp = MINIMUM(end - start, *lengthp);
                memcpy(datap, start, *lengthp);
                s_connection->bc_stats.bs_nread += *lengthp;
            }
            else
            {
                *lengthp = 0;
            }
        }
        else if (s_mode == BTS_MODE__DATALOGGING)
        {
            if (s_connection->bc_readnr_buffer_size > offset)
            {
                *lengthp = MINIMUM(
                    s_connection->bc_readnr_buffer_size - offset, *lengthp
                );
                memcpy(datap, &s_connection->bc_readnr_buffer[offset], *lengthp);
                s_connection->bc_stats.bs_nwritten += *lengthp;
            }
            else
            {
                *lengthp = 0;
            }
            bts_attribute_log(GATT_OPERATION__READ, datap, *lengthp, offset);
        }
        break;

    case GH_BTS__CHARACTERISTIC_CONFIG_DESCRIPTOR:
        if (offset)
        {
            return LTRX_GATT_STATUS__INVALID_OFFSET;
        }
        if (*lengthp < sizeof(s_connection->bc_cccd))
        {
            return LTRX_GATT_STATUS__INVALID_ATTRIBUTE_LENGTH;
        }
        memcpy(datap, &s_connection->bc_cccd, sizeof(s_connection->bc_cccd));
        *lengthp = sizeof(s_connection->bc_cccd);
        break;

    default:
        return LTRX_GATT_STATUS__INVALID_HANDLE;
    }
    return LTRX_GATT_STATUS__SUCCESS;
}

static int bts_attribute_write(
    uint16_t handle, uint16_t offset, uint8_t *datap, uint16_t length
)
{
    TLOG(
        TLOG_SEVERITY_LEVEL__DEBUG,
        "Write hnd: 0x%X  off: %u  len: %u",
        handle, offset, length
    );
    enum ltrx_gatt_status rc = LTRX_GATT_STATUS__INVALID_HANDLE;
    switch (handle)
    {
    case GH_BTS__CHARACTERISTIC_TEST_READ_WRITE_VALUE:
        if (s_mode == BTS_MODE__WRITEBACK)
        {
            if (offset >= s_connection->bc_write_buffer_size)
            {
                return LTRX_GATT_STATUS__INVALID_OFFSET;
            }
            if (offset + length > s_connection->bc_write_buffer_size)
            {
                return LTRX_GATT_STATUS__INVALID_ATTRIBUTE_LENGTH;
            }
            memcpy(
                &s_connection->bc_write_buffer[offset],
                datap,
                length
            );
            s_connection->bc_read_index = offset + length;
            s_connection->bc_stats.bs_nwritten += length;
        }
        else if (s_mode == BTS_MODE__DATALOGGING)
        {
            bts_attribute_log(GATT_OPERATION__WRITE, datap, length, offset);
            s_connection->bc_stats.bs_nwritten += length;
        }
        s_respond = true;
        ltrx_trigger_signal(&s_trigger);
        rc = LTRX_GATT_STATUS__PENDING;
        break;

    case GH_BTS__CHARACTERISTIC_TEST_READ_WRITE_NR_VALUE:
        if (s_mode == BTS_MODE__WRITEBACK)
        {
            if (offset >= s_connection->bc_writenr_buffer_size)
            {
                return LTRX_GATT_STATUS__INVALID_OFFSET;
            }
            if (offset + length > s_connection->bc_writenr_buffer_size)
            {
                return LTRX_GATT_STATUS__INVALID_ATTRIBUTE_LENGTH;
            }
            memcpy(
                &s_connection->bc_writenr_buffer[offset],
                datap,
                length
            );
            s_connection->bc_readnr_index = offset + length;
            s_connection->bc_stats.bs_nwritten += length;
        }
        else if (s_mode == BTS_MODE__DATALOGGING)
        {
            bts_attribute_log(GATT_OPERATION__WRITE, datap, length, offset);
            s_connection->bc_stats.bs_nwritten += length;
        }
        rc = LTRX_GATT_STATUS__SUCCESS;
        break;

    case GH_BTS__CHARACTERISTIC_CONFIG_DESCRIPTOR:
        if (length < sizeof(uint16_t))
        {
            rc = LTRX_GATT_STATUS__INVALID_ATTRIBUTE_LENGTH;
        }
        else
        {
#pragma GCC diagnostic push
#pragma GCC diagnostic ignored "-Wcast-align"
            uint16_t cccd = *((uint16_t *)datap);
#pragma GCC diagnostic pop
            s_connection->bc_cccd = cccd;
            rc = LTRX_GATT_STATUS__SUCCESS;
        }
        break;

    default:
        break;
    }
    return rc;
}

static int bts_attribute_write_prepare(
    uint16_t handle, uint16_t offset, uint8_t *datap, uint16_t length
)
{
    TLOG(
        TLOG_SEVERITY_LEVEL__DEBUG,
        "Write prepare hnd: 0x%X  off: %u  len: %u",
        handle, offset, length
    );
    if (
        handle != GH_BTS__CHARACTERISTIC_TEST_READ_WRITE_VALUE &&
        handle != GH_BTS__CHARACTERISTIC_TEST_READ_WRITE_NR_VALUE
    )
    {
        return LTRX_GATT_STATUS__INVALID_HANDLE;
    }
    struct bts_connection *bc = s_connection;
    if (bc->bc_prepared_write_count == ARRAY_SIZE(bc->bc_prepared_writes))
    {
        return LTRX_GATT_STATUS__PREPARE_QUE_FULL;
    }
    if (s_mode == BTS_MODE__WRITEBACK)
    {
        if (handle == GH_BTS__CHARACTERISTIC_TEST_READ_WRITE_VALUE)
        {
            if (offset >= bc->bc_write_buffer_size)
            {
                return LTRX_GATT_STATUS__INVALID_OFFSET;
            }
            if (offset + length > bc->bc_write_buffer_size)
            {
                return LTRX_GATT_STATUS__INVALID_ATTRIBUTE_LENGTH;
            }
        }
        else
        {
            if (offset >= bc->bc_writenr_buffer_size)
            {
                return LTRX_GATT_STATUS__INVALID_OFFSET;
            }
            if (offset + length > bc->bc_writenr_buffer_size)
            {
                return LTRX_GATT_STATUS__INVALID_ATTRIBUTE_LENGTH;
            }
        }
    }
    struct bts_prepared_write *bpw = &bc->bc_prepared_writes[
        bc->bc_prepared_write_count
    ];
    bpw->bpw_handle = handle;
    bpw->bpw_data = MALLOC(length);
    if (! bpw->bpw_data)
    {
        return LTRX_GATT_STATUS__INSUFFICIENT_RESOURCES;
    }
    bpw->bpw_length = length;
    bpw->bpw_offset = offset;
    memcpy(bpw->bpw_data, datap, length);
    bc->bc_prepared_write_count++;
    return LTRX_GATT_STATUS__SUCCESS;
}

static int bts_attribute_write_execute(bool proceed)
{
    TLOG(TLOG_SEVERITY_LEVEL__DEBUG, "Write %s", proceed ? "execute" : "cancel");
    struct bts_connection *bc = s_connection;
    if (bc->bc_prepared_write_count == 0)
    {
        return LTRX_GATT_STATUS__WRONG_STATE;
    }
    if (proceed)
    {
        for (uint8_t i = 0; i < bc->bc_prepared_write_count; ++i)
        {
            struct bts_prepared_write *bpw = &bc->bc_prepared_writes[i];
            if (s_mode == BTS_MODE__WRITEBACK)
            {
                memcpy(
                    (
                        bpw->bpw_handle == GH_BTS__CHARACTERISTIC_TEST_READ_WRITE_VALUE
                            ? &bc->bc_write_buffer[bpw->bpw_offset]
                            : &bc->bc_writenr_buffer[bpw->bpw_offset]
                    ),
                    bpw->bpw_data,
                    bpw->bpw_length
                );
            }
            else if (s_mode == BTS_MODE__DATALOGGING)
            {
                bts_attribute_log(
                    GATT_OPERATION__WRITE_EXECUTE,
                    bpw->bpw_data,
                    bpw->bpw_length,
                    bpw->bpw_offset
                );
            }
            bc->bc_stats.bs_nwritten += bpw->bpw_length;
            if (bpw->bpw_handle == GH_BTS__CHARACTERISTIC_TEST_READ_WRITE_VALUE)
            {
                bc->bc_read_index = MAXIMUM(
                    bc->bc_read_index, bpw->bpw_offset + bpw->bpw_length
                );
            }
            else
            {
                bc->bc_readnr_index = MAXIMUM(
                    bc->bc_readnr_index, bpw->bpw_offset + bpw->bpw_length
                );
            }
        }
    }
    for (uint8_t i = 0; i < bc->bc_prepared_write_count; ++i)
    {
        struct bts_prepared_write *bpw = &bc->bc_prepared_writes[i];
        ltrx_free(bpw->bpw_data);
        bpw->bpw_data = NULL;
        bpw->bpw_length = 0;
        bpw->bpw_offset = 0;
    }
    bc->bc_prepared_write_count = 0;
    return LTRX_GATT_STATUS__SUCCESS;
}

static int bts_attribute_mtu(
    uint16_t size
)
{
    if (size <= BTS_MTU_SIZE_MAX)
    {
        s_connection->bc_mtu_size = size;
        return LTRX_GATT_STATUS__SUCCESS;
    }
    return LTRX_GATT_STATUS__ILLEGAL_PARAMETER;
}

static void bts_attribute_notify(struct bts_connection *bc)
{
    struct bts_status status;
    status.time = ltrx_time_get();
    bts_get_stats(&status.stats);
    ltrx_gatt_send_notification(
        bc->bc_connection_id,
        GH_BTS__CHARACTERISTIC_CONFIG_DESCRIPTOR,
        (uint8_t *)&status,
        sizeof(status)
    );
}

static void bts_attribute_indicate(struct bts_connection *bc)
{
    struct bts_status status;
    status.time = ltrx_time_get();
    bts_get_stats(&status.stats);
    ltrx_gatt_send_indication(
        bc->bc_connection_id,
        GH_BTS__CHARACTERISTIC_CONFIG_DESCRIPTOR,
        (uint8_t *)&status,
        sizeof(status)
    );
}

static int bts_attribute_confirmation(
    uint16_t handle
)
{
    TLOG(TLOG_SEVERITY_LEVEL__DEBUG, "Confirm hnd 0x%X", handle);
    return LTRX_GATT_STATUS__SUCCESS;
}

static void bts_thread(void *arg)
{
    (void)arg;
    TLOG(
        TLOG_SEVERITY_LEVEL__INFORMATIONAL, BTS_SERVICE_NAME " thread started."
    );

    ltrx_trigger_create(&s_trigger, "");
    while (s_run_thread)
    {
        ltrx_trigger_clear(&s_trigger);
        bool rc = LTRX_TRIGGER_WAIT(&s_trigger, 10 * 1000);

        if (s_run_thread && s_connection)
        {
            if (s_respond)
            {
                ltrx_gatt_send_response(
                    s_connection->bc_connection_id,
                    LTRX_GATT_STATUS__SUCCESS,
                    GH_BTS__CHARACTERISTIC_TEST_READ_WRITE_VALUE,
                    NULL, 0, 0
                );
                s_respond = false;
            }
            else if (! rc)
            {
                if (s_connection->bc_cccd & LTRX_GATT_CLIENT_CONFIG__NOTIFICATION)
                {
                    bts_attribute_notify(s_connection);
                }
                if (s_connection->bc_cccd & LTRX_GATT_CLIENT_CONFIG__INDICATION)
                {
                    bts_attribute_indicate(s_connection);
                }
            }
        }
    }
    ltrx_trigger_destroy(&s_trigger);
    s_thread = NULL;
    TLOG(
        TLOG_SEVERITY_LEVEL__INFORMATIONAL, BTS_SERVICE_NAME " thread exit."
    );
}

static const struct ltrx_gatt_server s_server =
{
    .lgs_service_name = BTS_SERVICE_NAME,
    .lgs_service_count = 1,
    .lgs_service_array = { &s_service, },
    .lgs_connection = bts_connection,
    .lgs_attribute_request_mtu = bts_attribute_mtu,
    .lgs_attribute_request_write_execute = bts_attribute_write_execute,
    .lgs_attribute_request_read = bts_attribute_read,
    .lgs_attribute_request_write = bts_attribute_write,
    .lgs_attribute_request_write_prepare = bts_attribute_write_prepare,
    .lgs_attribute_request_confirmation = bts_attribute_confirmation,
};

bool bts_start(void)
{
    if (ltrx_bluetooth_get_state() == LTRX_BLUETOOTH_STATE__DISABLED)
    {
        return false;
    }
    if (! s_registered)
    {
        s_run_thread = true;
        s_thread = ltrx_thread_create(
            s_server.lgs_service_name,
            bts_thread,
            NULL,
            STACK_SIZE_GREEN_FROM_MAX_OBSERVED_STACK_USED(3100)
        );
        enum ltrx_bluetooth_state lbs;
        do {
            lbs = ltrx_bluetooth_get_state();
            if (lbs != LTRX_BLUETOOTH_STATE__RUNNING)
            {
                ltrx_thread_sleep(1000);
            }

        } while (lbs != LTRX_BLUETOOTH_STATE__RUNNING);
        s_registered = ltrx_gatt_register_server(&s_server);
        if (! s_registered)
        {
            TLOG(TLOG_SEVERITY_LEVEL__ERROR, "Failed to register server.");
            return false;
        }
    }
    return true;
}

bool bts_stop(void)
{
    if (s_registered)
    {
        ltrx_preemption_block();
        if (s_connection)
        {
            uint16_t cid = s_connection->bc_connection_id;
            ltrx_preemption_unblock();
            ltrx_gatt_disconnect(cid);
        }
        else
        {
            ltrx_preemption_unblock();
        }
        ltrx_gatt_unregister_server(&s_server);
        s_registered = false;
        s_run_thread = false;
        ltrx_trigger_signal(&s_trigger);
        do { ltrx_thread_sleep(100); } while (s_thread != NULL);
    }
    return true;
}

bool bts_is_running(void)
{
    return s_registered;
}

void bts_mode_set(uint8_t m)
{
    if (m <= BTS_MODE__DATALOGGING && m != s_mode)
    {
        s_mode = m;
        if (s_connection)
        {
            bts_mode_configure(s_mode, s_connection);
        }
    }
}

uint8_t bts_mode_get(void)
{
    return s_mode;
}

bool bts_is_client_connected(void)
{
    return s_connection != NULL;
}

void bts_get_stats(struct bts_stats *bss)
{
    ltrx_preemption_block();
    memset(bss, 0, sizeof(*bss));
    if (s_connection)
    {
        memcpy(bss, &s_connection->bc_stats, sizeof(*bss));
        bss->bs_connection_id = s_connection->bc_connection_id;
        bss->bs_mtu = s_connection->bc_mtu_size;
        bss->bs_queued_writes = s_connection->bc_prepared_write_count;
        for (uint8_t i = 0; i < s_connection->bc_prepared_write_count; ++i)
        {
            bss->bs_queued_write_bytes += s_connection->bc_prepared_writes[i].bpw_length;
        }
    }
    ltrx_preemption_unblock();
}


