/*****************************************************************************/
/*                              Legal                                        */
/*****************************************************************************/

/*
** Copyright ©2015-2025, Lantronix, Inc. All Rights Reserved.
** By using this software, you are agreeing to the terms of the Software
** Development Kit (SDK) License Agreement included in the distribution package
** for this software (the “License Agreement”).
** Under the License Agreement, this software may be used solely to create
** custom applications for use on the Lantronix xPico Wi-Fi product.
** THIS SOFTWARE AND ANY ACCOMPANYING DOCUMENTATION IS PROVIDED "AS IS".
** LANTRONIX SPECIFICALLY DISCLAIMS ANY WARRANTIES, INCLUDING, BUT NOT LIMITED
** TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY, NON-INFRINGEMENT AND FITNESS
** FOR A PARTICULAR PURPOSE.
** LANTRONIX HAS NO OBLIGATION TO PROVIDE MAINTENANCE, SUPPORT, UPDATES,
** ENHANCEMENTS, OR MODIFICATIONS TO THIS SOFTWARE.
** IN NO EVENT SHALL LANTRONIX BE LIABLE TO ANY PARTY FOR DIRECT, INDIRECT,
** SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES, INCLUDING LOST PROFITS,
** ARISING OUT OF THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF
** LANTRONIX HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
*/

/*****************************************************************************/
/*                           Documentation                                   */
/*****************************************************************************/

/*!
** \addtogroup example
** @{
*/

/*!
** \defgroup ble_gatt_server_demo ble_gatt_server_demo
** @{
**
** The \b ble_gatt_server_demo module is the most basic Bluetooth Low Energy (BLE)
** demo. Creates a connection that demonstrates reading and writing data.
** Demonstrates GATT database usage and device configuration. Requires the use
** of LightBlue found on the Google Play and Apple App Stores. This app can be
** used to connect to the xPico device using Bluetooth. Will not support use of
** the Lantronix Gateway Provisioning App. Use the TLOG to confirm the
** transmission of data. The default Bluetooth SSID is "BLE Gatt Demo".
**
** Build it from project "bleGattServerDemo".
*/

/*****************************************************************************/
/*                             Includes                                      */
/*****************************************************************************/

#include <stdio.h>
#include <string.h>

#include "ble_gatt_server_demo_module_defs.h" /* Automatically generated by make. */
#include "ble_gatt_server_demo.h"
#include "bluetooth_module_libs.h"
#include "ltrx_gatt.h"
#include "ltrx_gatt_server.h"
#include "ltrx_tlog.h" /* Delivered with SDK. */


#define UUID_DESCRIPTOR_CLIENT_CHARACTERISTIC_CONFIGURATION 0x2902 // FIXME

/****************************************************************************/
/*                             Data Structures                              */
/****************************************************************************/

struct ble_gatt_server_demo_state
{
    uint16_t conn_id;                  /* connection ID referenced by the stack */
    uint8_t num_to_write;             /* num msgs to send, incr on each button intr */
    uint8_t flag_indication_sent;     /* indicates waiting for ack/cfm */
    uint8_t flag_stay_connected;      /* stay connected or disconnect after all messages are sent */
    uint8_t battery_level;            /* dummy battery level */
} ;

struct ble_gatt_server_demo_host_info
{
    uint16_t characteristic_client_configuration;  /* Current value of the client configuration descriptor */
};

struct ble_attribute
{
    uint16_t handle;
    uint16_t attr_len;
    void *p_attr;
};

enum ble_gatt_server_demo_handle
{
    BGSD_HANDLE__SERVICE = LTRX_GATT_HANDLE_BASE + 0x200,
    BGSD_HANDLE__CHAR_NOTIFY, /* characteristic handle */
    BGSD_HANDLE__CHAR_NOTIFY_VAL, /* char value handle */
    BGSD_HANDLE__CHAR_CFG_DESC, /* charconfig desc handle */
    BGSD_HANDLE__CHAR_BLINK, /* characteristic handle */
    BGSD_HANDLE__CHAR_BLINK_VAL, /* char value handle */
    BGSD_HANDLE__CHAR_LONG_MSG, /* characteristic handle */
    BGSD_HANDLE__CHAR_LONG_MSG_VAL, /* long  char value handle */
    HANDLE_BLE_GATT_DEMO_BATTERY_SERVICE, /* service handle */
    BGSD_HANDLE__BATTERY_CHAR_LEVEL, /* characteristic handle */
    BGSD_HANDLE__BATTERY_CHAR_LEVEL_VAL, /* char value handle */
};


/****************************************************************************/
/*                                 Globals                                  */
/****************************************************************************/

const struct bluetooth_external_functions *g_bluetoothExternalFunctionEntry_pointer;


/****************************************************************************/
/*                                  Locals                                  */
/****************************************************************************/

#pragma GCC diagnostic push
#pragma GCC diagnostic ignored "-Wdiscarded-qualifiers"

static const struct ltrx_gatt_server_descriptor s_descriptor =
{
    .uuid = {
        UUID_LENGTH_16, { UUID_DESCRIPTOR_CLIENT_CHARACTERISTIC_CONFIGURATION }
    },
    .handle = BGSD_HANDLE__CHAR_CFG_DESC,
    .permissions = GATT_DB_PERMISSION__READABLE | GATT_DB_PERMISSION__WRITE_REQ,
};

static const struct ltrx_gatt_server_characteristic s_characteristic[] =
{
    {
        .uuid = {
            UUID_LENGTH_128,
            { .id128 = { UUID_BLE_GATT_DEMO_CHARACTERISTIC_NOTIFY } }
        },
        .handle = BGSD_HANDLE__CHAR_NOTIFY,
        .value_handle = BGSD_HANDLE__CHAR_NOTIFY_VAL,
        .properties = (
            GATT_DB_PROPERTY__READ | GATT_DB_PROPERTY__NOTIFY | GATT_DB_PROPERTY__INDICATE
        ),
        .permissions = GATT_DB_PERMISSION__READABLE,
        .descriptor_count = 1,
        .descriptor_array = &s_descriptor,
    },
    {
        .uuid = {
            UUID_LENGTH_128,
            { .id128 = { UUID_BLE_GATT_DEMO_CHARACTERISTIC_CONFIG } }
        },
        .handle = BGSD_HANDLE__CHAR_BLINK,
        .value_handle = BGSD_HANDLE__CHAR_BLINK_VAL,
        .properties = GATT_DB_PROPERTY__READ | GATT_DB_PROPERTY__WRITE,
        .permissions = (
            GATT_DB_PERMISSION__READABLE | GATT_DB_PERMISSION__WRITE_CMD | GATT_DB_PERMISSION__WRITE_REQ
        ),
        .descriptor_count = 0,
        .descriptor_array = NULL,
    },
};

static const struct ltrx_gatt_server_service s_service =
{
    .uuid = {
        UUID_LENGTH_128,
        { .id128 = { UUID_BLE_GATT_DEMO_SERVICE } }
    },
    .handle = BGSD_HANDLE__SERVICE,
    .advertise = true,
    .characteristic_count = 2,
    .characteristic_array = s_characteristic,
};
#pragma GCC diagnostic pop

static struct ble_gatt_server_demo_state s_state;

static struct ble_gatt_server_demo_host_info s_hostinfo;

static char s_notify_value[] = { '0' };

static const struct ble_attribute s_attributes[] =
{
    {
        BGSD_HANDLE__CHAR_NOTIFY_VAL,
        sizeof(s_notify_value),
        s_notify_value
    },
    {
        BGSD_HANDLE__CHAR_CFG_DESC,
        sizeof(s_hostinfo.characteristic_client_configuration),
        (void *)&s_hostinfo.characteristic_client_configuration
    },
    {
        BGSD_HANDLE__BATTERY_CHAR_LEVEL_VAL,
        sizeof(s_state.battery_level),
        &s_state.battery_level
    },
};


/****************************************************************************/
/*                           Function Definitions                           */
/****************************************************************************/

static void bgsd_connection(uint16_t connection_id, bool connected)
{
    if (connected)
    {
        s_state.conn_id = connection_id;
        s_hostinfo.characteristic_client_configuration = 0;
    }
    else
    {
        s_state.conn_id = 0;
    }
}

static const struct ble_attribute *bgsd_get_attribute(uint16_t handle)
{
    for (uint8_t i = 0; i < ARRAY_SIZE(s_attributes); i++)
    {
        if (s_attributes[i].handle == handle)
        {
            return (&s_attributes[i]);
        }
    }
    return NULL;
}

static int bgsd_attribute_read(
    uint16_t handle, uint16_t offset, uint8_t *datap, uint16_t *lengthp
)
{
    TLOG(
        TLOG_SEVERITY_LEVEL__DEBUG,
        "%s(0x%X, %u, %p, %p)",
        __func__, handle, offset, datap, lengthp
    );
    const struct ble_attribute *ba = bgsd_get_attribute(handle);
    if (ba == NULL)
    {
        return LTRX_GATT_STATUS__INVALID_HANDLE;
    }
    if (
        handle == BGSD_HANDLE__BATTERY_CHAR_LEVEL_VAL &&
        s_state.battery_level++ > 5
   )
    {
        s_state.battery_level = 0;
    }
    uint16_t length = ba->attr_len;
    if (offset > ba->attr_len)
    {
        length = 0;
    }
    if (length != 0)
    {
        uint8_t *from;
        uint16_t to_copy = length - offset;
        if (to_copy > *lengthp)
        {
            to_copy = *lengthp;
        }
        from = ((uint8_t *)ba->p_attr) + offset;
        *lengthp = to_copy;
        memcpy(datap, from, to_copy);
        TLOG(
            TLOG_SEVERITY_LEVEL__INFORMATIONAL,
            "Returning %u bytes: %*B",
            to_copy, to_copy, datap
        );
    }
    return LTRX_GATT_STATUS__SUCCESS;
}

static void ble_gatt_demo_send_message(void)
{
    if (s_hostinfo.characteristic_client_configuration != 0)
    {
        if (s_hostinfo.characteristic_client_configuration & 0x0001)
        {
            uint8_t *p_attr = (uint8_t *)s_notify_value;
            ltrx_gatt_send_notification(
                s_state.conn_id,
                BGSD_HANDLE__CHAR_NOTIFY_VAL,
                p_attr,
                sizeof(*p_attr)
            );
        }
        else if (! s_state.flag_indication_sent)
        {
            uint8_t *p_attr = (uint8_t *)s_notify_value;
            s_state.flag_indication_sent = true;
            ltrx_gatt_send_indication(
                s_state.conn_id,
                BGSD_HANDLE__CHAR_NOTIFY_VAL,
                p_attr,
                sizeof(*p_attr)
            );
        }
    }
}

static int bgsd_attribute_confirmation(uint16_t handle)
{
    TLOG(TLOG_SEVERITY_LEVEL__DEBUG, "%s(0x%X)", __func__, handle);
    (void)handle;
    if (! s_state.flag_indication_sent)
    {
        return LTRX_GATT_STATUS__SUCCESS;
    }
    s_state.flag_indication_sent = 0;
    if (s_state.num_to_write)
    {
        s_state.num_to_write--;
        ble_gatt_demo_send_message();
    }
    return LTRX_GATT_STATUS__SUCCESS;
}

static void bgsd_increment_notify_value(void)
{
    uint16_t last_byte = sizeof(s_notify_value) - 1 ;
    char c = s_notify_value[last_byte];
    c++;
    if ((c < '0') || (c > '9'))
    {
        c = '0';
    }
    s_notify_value[last_byte] = c;
}

static int bgsd_attribute_write(
    uint16_t handle, uint16_t offset, uint8_t *datap, uint16_t length
)
{
    TLOG(
        TLOG_SEVERITY_LEVEL__DEBUG,
        "%s(0x%X, %u, %p, %u)",
        __func__, handle, offset, datap, length
    );
    if (handle == BGSD_HANDLE__CHAR_CFG_DESC)
    {
        if (length != 2)
        {
            return LTRX_GATT_STATUS__INVALID_ATTRIBUTE_LENGTH;
        }
        s_hostinfo.characteristic_client_configuration = (
            datap[0] | (datap[1] << 8)
        );
    }
    else if (handle == BGSD_HANDLE__CHAR_BLINK_VAL)
    {
        bgsd_increment_notify_value();
        TLOG(TLOG_SEVERITY_LEVEL__INFORMATIONAL, "Data written: %04X", datap[0]);
        /*****************************************
         * Update LED to blink through GPIO here *
         *****************************************/
    }
    else
    {
        return LTRX_GATT_STATUS__INVALID_HANDLE;
    }
    return LTRX_GATT_STATUS__SUCCESS;
}

#pragma GCC diagnostic push
#pragma GCC diagnostic ignored "-Wdiscarded-qualifiers"
static const struct ltrx_gatt_server s_server =
{
    .lgs_service_name = "BLE GATT Demo",
    .lgs_service_count = 1,
    .lgs_service_array = { &s_service },
    .lgs_connection = bgsd_connection,
    .lgs_attribute_request_mtu = NULL,
    .lgs_attribute_request_write_execute = NULL,
    .lgs_attribute_request_read = bgsd_attribute_read,
    .lgs_attribute_request_write = bgsd_attribute_write,
    .lgs_attribute_request_write_prepare = NULL,
    .lgs_attribute_request_confirmation = bgsd_attribute_confirmation,
};
#pragma GCC diagnostic pop

static void bgsd_thread(void *arg)
{
    (void)arg;

    while (ltrx_bluetooth_get_state() != LTRX_BLUETOOTH_STATE__RUNNING)
    {
        ltrx_thread_sleep(1000);
    }
    if (! ltrx_gatt_register_server(&s_server))
    {
        TLOG(TLOG_SEVERITY_LEVEL__ERROR, "Failed to register server.");
    }
}

void ble_gatt_server_demo_start(void)
{
    g_bluetoothExternalFunctionEntry_pointer = ltrx_module_functions_lookup(
        "bluetooth"
    );
    ltrx_thread_create(
        s_server.lgs_service_name,
        bgsd_thread,
        NULL,
        STACK_SIZE_GREEN_FROM_MAX_OBSERVED_STACK_USED(600)
    );
}

/*! @} End of Group. */
/*! @} End of Group. */
