/*
 * Copyright 2019-2025, Lantronix, Inc. All Rights Reserved.
 *
 * This is UNPUBLISHED PROPRIETARY SOURCE CODE of Lantronix, Inc.;
 * the contents of this file may not be disclosed to third parties, copied
 * or duplicated in any form, in whole or in part, without the prior
 * written permission of Lantronix, Inc.
 */

/****************************************************************************/
/*                                 Includes                                 */
/****************************************************************************/

#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include "ble_strings.h"
#include "gatt_definitions.h"
#include "ltrx_bluetooth.h"
#include "ltrx_gatt.h"
#include "wiced_bt_uuid.h"


/****************************************************************************/
/*                                 Globals                                  */
/****************************************************************************/

const char * const g_bt_transport_desc[] =
{
    "NONE (0)",
    "BT_TRANSPORT_BR_EDR",
    "BT_TRANSPORT_LE"
};

const char * const g_ble_addr_type_desc[] = {
    "PUBLIC",
    "RANDOM",
    "PUBLIC_ID",
    "RANDOM_ID",
};

const char * const g_gatt_discovery_type_desc[] =
{
    "<INVALID>",
    "DISCOVER_SERVICES_ALL",
    "DISCOVER_SERVICES_BY_UUID",
    "DISCOVER_INCLUDED_SERVICES",
    "DISCOVER_CHARACTERISTICS",
    "DISCOVER_CHARACTERISTIC_DESCRIPTORS",
};

/****************************************************************************/
/*                           Function Definitions                           */
/****************************************************************************/

const char *str_ble_uuid_service(uint16_t uuid)
{
    switch (uuid)
    {
    case UUID_SERVICE_GAP: return "GAP";
    case UUID_SERVICE_GATT: return "GATT";
    case UUID_SERVICE_IMMEDIATE_ALERT: return "IMMEDIATE_ALERT";
    case UUID_SERVICE_LINK_LOSS: return "LINK_LOSS";
    case UUID_SERVICE_TX_POWER: return "TX_POWER";
    case UUID_SERVICE_CURRENT_TIME: return "CURRENT_TIME";
    case UUID_SERVICE_REFERENCE_TIME_UPDATE: return "REFERENCE_TIME_UPDATE";
    case UUID_SERVICE_DST_CHANGE: return "DST_CHANGE";
    case UUID_SERVICE_GLUCOSE: return "GLUCOSE";
    case UUID_SERVICE_HEALTH_THERMOMETER: return "HEALTH_THERMOMETER";
    case UUID_SERVICE_DEVICE_INFORMATION: return "DEVICE_INFO";
    case UUID_SERVICE_NETWORK_AVAILABILITY: return "NETWORK_AVAILABILITY";
    case UUID_SERVICE_WATCHDOG: return "WATCHDOG";
    case UUID_SERVICE_HEART_RATE: return "HEART_RATE";
    case UUID_SERVICE_PHONE_ALERT_STATUS: return "PHONE_ALERT_STATUS";
    case UUID_SERVICE_BATTERY: return "BATTERY";
    case UUID_SERVICE_BLOOD_PRESSURE: return "BLOOD_PRESSURE";
    case UUID_SERVICE_ALERT_NOTIFICATION: return "ALERT_NOTIFICATION";
    case UUID_SERVICE_HID: return "HID";
    case UUID_SERVICE_SCAN_PARAMETERS: return "SCAN_PARAMETERS";
    case UUID_SERVICE_RSC: return "RSC";
    case UUID_SERVICE_AUTOMATION_IO: return "AUTOMATION_IO";
    case UUID_SERVICE_CSC: return "CSC";
    case UUID_SERVICE_CYCLING_POWER: return "CYCLING_POWER";
    case UUID_SERVICE_LOCATION_NAVIGATION: return "LOCATION_NAVIGATION";
    case UUID_SERVICE_ENVIRONMENTAL_SENSING: return "ENVIRONMENTAL_SENSING";
    case UUID_SERVICE_BODY_COMPOSITION: return "BODY_COMPOSITION";
    case UUID_SERVICE_USER_DATA: return "USER_DATA";
    case UUID_SERVICE_WEIGHT_SCALE: return "WEIGHT_SCALE";
    case UUID_SERVICE_BOND_MANAGEMENT: return "BOND_MANAGEMENT";
    case UUID_SERVICE_CONTINUOUS_GLUCOSE_MONITORING: return "CONTINUOUS_GLUCOSE_MONITORING";
    case UUID_SERVICE_INTERNET_PROTOCOL_SUPPORT: return "INTERNET_PROTOCOL_SUPPORT";
    case UUID_SERVICE_INDOOR_POSITIONING: return "INDOOR_POSITIONING";
    case UUID_SERVICE_PULSE_OXIMETER: return "PULSE_OXIMETER";
    case UUID_SERVICE_HTTP_PROXY: return "HTTP_PROXY";
    default: return "";
    }
}

const char *str_ble_uuid_characteristic(uint16_t uuid)
{
    switch (uuid)
    {
    case UUID_CHARACTERISTIC_DEVICE_NAME: return "DEVICE_NAME";
    case UUID_CHARACTERISTIC_APPEARANCE: return "APPEARANCE";
    case UUID_CHARACTERISTIC_PERIPHERAL_PRIVACY_FLAG: return "PERIPHERAL_PRIVACY_FLAG";
    case UUID_CHARACTERISTIC_RECONNECTION_ADDRESS: return "RECONNECTION_ADDRESS";
    case UUID_CHARACTERISTIC_PERIPHERAL_PREFERRED_CONNECTION_PARAMETERS: return "PREFERRED_CONNECTION_PARAMETERS";
    case UUID_CHARACTERISTIC_SERVICE_CHANGED: return "SERVICE_CHANGED";
    case UUID_CHARACTERISTIC_ALERT_LEVEL: return "ALERT_LEVEL";
    case UUID_CHARACTERISTIC_TX_POWER_LEVEL: return "TX_POWER_LEVEL";
    case UUID_CHARACTERISTIC_DATE_TIME: return "DATE_TIME";
    case UUID_CHARACTERISTIC_DAY_OF_WEEK: return "DAY_OF_WEEK";
    case UUID_CHARACTERISTIC_TIME: return "TIME";
    case UUID_CHARACTERISTIC_EXACT_TIME_100: return "EXACT_TIME_100";
    case UUID_CHARACTERISTIC_EXACT_TIME_256: return "EXACT_TIME_256";
    case UUID_CHARACTERISTIC_DAYLIGHT_SAVING_TIME: return "DAYLIGHT_SAVING_TIME";
    case UUID_CHARACTERISTIC_TIME_ZONE: return "TIME_ZONE";
    case UUID_CHARACTERISTIC_LOCAL_TIME_INFORMATION: return "LOCAL_TIME_INFORMATION";
    case UUID_CHARACTERISTIC_SECONDARY_TIME_ZONE: return "SECONDARY_TIME_ZONE";
    case UUID_CHARACTERISTIC_TIME_WITH_DST: return "TIME_WITH_DST";
    case UUID_CHARACTERISTIC_TIME_ACCURACY: return "TIME_ACCURACY";
    case UUID_CHARACTERISTIC_TIME_SOURCE: return "TIME_SOURCE";
    case UUID_CHARACTERISTIC_REFERENCE_TIME_INFORMATION: return "REFERENCE_TIME_INFORMATION";
    case UUID_CHARACTERISTIC_TIME_BROADCAST: return "TIME_BROADCAST";
    case UUID_CHARACTERISTIC_TIME_UPDATE_CONTROL_POINT: return "UPDATE_CONTROL_POINT";
    case UUID_CHARACTERISTIC_TIME_UPDATE_STATE: return "TIME_UPDATE_STATE";
    case UUID_CHARACTERISTIC_GLUCOSE_MEASUREMENT: return "GLUCOSE_MEASUREMENT";
    case UUID_CHARACTERISTIC_BATTERY_LEVEL: return "BATTERY_LEVEL";
    case UUID_CHARACTERISTIC_BATTERY_POWER_STATE: return "BATTERY_POWER_STATE";
    case UUID_CHARACTERISTIC_BATTERY_LEVEL_STATE: return "BATTERY_LEVEL_STATE";
    case UUID_CHARACTERISTIC_TEMPERATURE_MEASUREMENT: return "TEMPERATURE_MEASUREMENT";
    case UUID_CHARACTERISTIC_TEMPERATURE_TYPE: return "TEMPERATURE_TYPE";
    case UUID_CHARACTERISTIC_INTERMEDIATE_TEMPERATURE: return "INTERMEDIATE_TEMPERATURE";
    case UUID_CHARACTERISTIC_TEMPERATURE_CELSIUS: return "TEMPERATURE_CELSIUS";
    case UUID_CHARACTERISTIC_TEMPERATURE_FAHRENHEIT: return "TEMPERATURE_FAHRENHEIT";
    case UUID_CHARACTERISTIC_MEASUREMENT_INTERVAL: return "MEASUREMENT_INTERVAL";
    case UUID_CHARACTERISTIC_BOOT_KEYBOARD_INPUT_REPORT: return "BOOT_KEYBOARD_INPUT_REPORT";
    case UUID_CHARACTERISTIC_SYSTEM_ID: return "SYSTEM_ID";
    case UUID_CHARACTERISTIC_MODEL_NUMBER_STRING: return "MODEL_NUMBER_STRING";
    case UUID_CHARACTERISTIC_SERIAL_NUMBER_STRING: return "SERIAL_NUMBER_STRING";
    case UUID_CHARACTERISTIC_FIRMWARE_REVISION_STRING: return "FIRMWARE_REVISION_STRING";
    case UUID_CHARACTERISTIC_HARDWARE_REVISION_STRING: return "HARDWARE_REVISION_STRING";
    case UUID_CHARACTERISTIC_SOFTWARE_REVISION_STRING: return "SOFTWARE_REVISION_STRING";
    case UUID_CHARACTERISTIC_MANUFACTURER_NAME_STRING: return "MANUFACTURER_NAME_STRING";
    case UUID_CHARACTERISTIC_IEEE_11073_20601_REGULATORY_CERTIFICATION_DATA_LIST: return "IEEE_11073_20601_REGULATORY_CERTIFICATION_DATA_LIST";
    case UUID_CHARACTERISTIC_CURRENT_TIME: return "CURRENT_TIME";
    case UUID_CHARACTERISTIC_MAGNETIC_DECLINATION: return "MAGNETIC_DECLINATION";
    case UUID_CHARACTERISTIC_SCAN_REFRESH: return "SCAN_REFRESH";
    case UUID_CHARACTERISTIC_BOOT_KEYBOARD_OUTPUT_REPORT: return "BOOT_KEYBOARD_OUTPUT_REPORT";
    case UUID_CHARACTERISTIC_BOOT_MOUSE_INPUT_REPORT: return "BOOT_MOUSE_INPUT_REPORT";
    case UUID_CHARACTERISTIC_GLUCOSE_MEASUREMENT_CONTEXT: return "GLUCOSE_MEASUREMENT_CONTEXT";
    case UUID_CHARACTERISTIC_BLOOD_PRESSURE_MEASUREMENT: return "BLOOD_PRESSURE_MEASUREMENT";
    case UUID_CHARACTERISTIC_INTERMEDIATE_BLOOD_PRESSURE: return "INTERMEDIATE_BLOOD_PRESSURE";
    case UUID_CHARACTERISTIC_HEART_RATE_MEASUREMENT: return "HEART_RATE_MEASUREMENT";
    case UUID_CHARACTERISTIC_HEART_RATE_SENSOR_LOCATION: return "HEART_RATE_SENSOR_LOCATION";
    case UUID_CHARACTERISTIC_HEART_RATE_CONTROL_POINT: return "HEART_RATE_CONTROL_POINT";
    case UUID_CHARACTERISTIC_REMOVABLE: return "REMOVABLE";
    case UUID_CHARACTERISTIC_SERVICE_REQUIRED: return "SERVICE_REQUIRED";
    case UUID_CHARACTERISTIC_SCIENTIFIC_TEMPERATURE_CELSIUS: return "SCIENTIFIC_TEMPERATURE_CELSIUS";
    case UUID_CHARACTERISTIC_STRING: return "STRING";
    case UUID_CHARACTERISTIC_NETWORK_AVAILABILITY: return "NETWORK_AVAILABILITY";
    case UUID_CHARACTERISTIC_ALERT_STATUS: return "ALERT_STATUS";
    case UUID_CHARACTERISTIC_RINGER_CONTROL_POINT: return "RINGER_CONTROL_POINT";
    case UUID_CHARACTERISTIC_RINGER_SETTING: return "RINGER_SETTING";
    case UUID_CHARACTERISTIC_ALERT_CATEGORY_ID_BIT_MASK: return "ALERT_CATEGORY_ID_BIT_MASK";
    case UUID_CHARACTERISTIC_ALERT_CATEGORY_ID: return "ALERT_CATEGORY_ID";
    case UUID_CHARACTERISTIC_ALERT_NOTIFICATION_CONTROL_POINT: return "ALERT_NOTIFICATION_CONTROL_POINT";
    case UUID_CHARACTERISTIC_UNREAD_ALERT_STATUS: return "UNREAD_ALERT_STATUS";
    case UUID_CHARACTERISTIC_NEW_ALERT: return "NEW_ALERT";
    case UUID_CHARACTERISTIC_SUPPORTED_NEW_ALERT_CATEGORY: return "SUPPORTED_NEW_ALERT_CATEGORY";
    case UUID_CHARACTERISTIC_SUPPORTED_UNREAD_ALERT_CATEGORY: return "SUPPORTED_UNREAD_ALERT_CATEGORY";
    case UUID_CHARACTERISTIC_BLOOD_PRESSURE_FEATURE: return "BLOOD_PRESSURE_FEATURE";
    case UUID_CHARACTERISTIC_HID_INFORMATION: return "HID_INFORMATION";
    case UUID_CHARACTERISTIC_REPORT_MAP: return "REPORT_MAP";
    case UUID_CHARACTERISTIC_HID_CONTROL_POINT: return "HID_CONTROL_POINT";
    case UUID_CHARACTERISTIC_REPORT: return "REPORT";
    case UUID_CHARACTERISTIC_PROTOCOL_MODE: return "PROTOCOL_MODE";
    case UUID_CHARACTERISTIC_SCAN_INTERVAL_WINDOW: return "SCAN_INTERVAL_WINDOW";
    case UUID_CHARACTERISTIC_PNP_ID: return "PNP_ID";
    case UUID_CHARACTERISTIC_GLUCOSE_FEATURES: return "GLUCOSE_FEATURES";
    case UUID_CHARACTERISTIC_RECORD_ACCESS_CONTROL_POINT: return "RECORD_ACCESS_CONTROL_POINT";
    case UUID_CHARACTERISTIC_RSC_MEASUREMENT: return "RSC_MEASUREMENT";
    case UUID_CHARACTERISTIC_RSC_FEATURE: return "RSC_FEATURE";
    case UUID_CHARACTERISTIC_SC_CONTROL_POINT: return "SC_CONTROL_POINT";
    case UUID_CHARACTERISTIC_DIGITAL: return "DIGITAL";
    case UUID_CHARACTERISTIC_ANALOG: return "ANALOG";
    case UUID_CHARACTERISTIC_AGGREGATE_INPUT: return "AGGREGATE_INPUT";
    case UUID_CHARACTERISTIC_CSC_MEASUREMENT: return "CSC_MEASUREMENT";
    case UUID_CHARACTERISTIC_CSC_FEATURE: return "CSC_FEATURE";
    case UUID_CHARACTERISTIC_SENSOR_LOCATION: return "SENSOR_LOCATION";
    case UUID_CHARACTERISTIC_PLX_SPOT_CHECK_MEASUREMENT: return "PLX_SPOT_CHECK_MEASUREMENT";
    case UUID_CHARACTERISTIC_PLX_CONTINUOUS_MEASUREMENT: return "PLX_CONTINUOUS_MEASUREMENT";
    case UUID_CHARACTERISTIC_PLX_FEATURES: return "PLX_FEATURES";
    case UUID_CHARACTERISTIC_CYCLING_POWER_MEASUREMENT: return "CYCLING_POWER_MEASUREMENT";
    case UUID_CHARACTERISTIC_CYCLING_POWER_VECTOR: return "CYCLING_POWER_VECTOR";
    case UUID_CHARACTERISTIC_CYCLING_POWER_FEATURE: return "CYCLING_POWER_FEATURE";
    case UUID_CHARACTERISTIC_CYCLING_POWER_CONTROL_POINT: return "CYCLING_POWER_CONTROL_POINT";
    case UUID_CHARACTERISTIC_LOCATION_AND_SPEED: return "LOCATION_AND_SPEED";
    case UUID_CHARACTERISTIC_NAVIGATION: return "NAVIGATION";
    case UUID_CHARACTERISTIC_POSITION_QUALITY: return "POSITION_QUALITY";
    case UUID_CHARACTERISTIC_LN_FEATURE: return "LN_FEATURE";
    case UUID_CHARACTERISTIC_LN_CONTROL_POINT: return "LN_CONTROL_POINT";
    case UUID_CHARACTERISTIC_ELEVATION: return "ELEVATION";
    case UUID_CHARACTERISTIC_PRESSURE: return "PRESSURE";
    case UUID_CHARACTERISTIC_TEMPERATURE: return "TEMPERATURE";
    case UUID_CHARACTERISTIC_HUMIDITY: return "HUMIDITY";
    case UUID_CHARACTERISTIC_TRUE_WIND_SPEED: return "TRUE_WIND_SPEED";
    case UUID_CHARACTERISTIC_TRUE_WIND_DIRECTION: return "TRUE_WIND_DIRECTION";
    case UUID_CHARACTERISTIC_APPARENT_WIND_SPEED: return "APPARENT_WIND_SPEED";
    case UUID_CHARACTERISTIC_APPARENT_WIND_DIRECTION: return "APPARENT_WIND_DIRECTION";
    case UUID_CHARACTERISTIC_GUST_FACTOR: return "GUST_FACTOR";
    case UUID_CHARACTERISTIC_POLLEN_CONCENTRATION: return "POLLEN_CONCENTRATION";
    case UUID_CHARACTERISTIC_UV_INDEX: return "UV_INDEX";
    case UUID_CHARACTERISTIC_IRRADIANCE: return "IRRADIANCE";
    case UUID_CHARACTERISTIC_RAINFALL: return "RAINFALL";
    case UUID_CHARACTERISTIC_WIND_CHILL: return "WIND_CHILL";
    case UUID_CHARACTERISTIC_HEAT_INDEX: return "HEAT_INDEX";
    case UUID_CHARACTERISTIC_DEW_POINT: return "DEW_POINT";
    case UUID_CHARACTERISTIC_DESCRIPTOR_VALUE_CHANGED: return "DESCRIPTOR_VALUE_CHANGED";
    case UUID_CHARACTERISTIC_AEROBIC_HEART_RATE_LOWER_LIMIT: return "AEROBIC_HEART_RATE_LOWER_LIMIT";
    case UUID_CHARACTERISTIC_AEROBIC_THRESHOLD: return "AEROBIC_THRESHOLD";
    case UUID_CHARACTERISTIC_AGE: return "AGE";
    case UUID_CHARACTERISTIC_ANAEROBIC_HEART_RATE_LOWER_LIMIT: return "ANAEROBIC_HEART_RATE_LOWER_LIMIT";
    case UUID_CHARACTERISTIC_ANAEROBIC_HEART_RATE_UPPER_LIMIT: return "ANAEROBIC_HEART_RATE_UPPER_LIMIT";
    case UUID_CHARACTERISTIC_ANAEROBIC_THRESHOLD: return "ANAEROBIC_THRESHOLD";
    case UUID_CHARACTERISTIC_AEROBIC_HEART_RATE_UPPER_LIMIT: return "AEROBIC_HEART_RATE_UPPER_LIMIT";
    case UUID_CHARACTERISTIC_DATE_OF_BIRTH: return "DATE_OF_BIRTH";
    case UUID_CHARACTERISTIC_DATE_OF_THRESHOLD_ASSESSMENT: return "DATE_OF_THRESHOLD_ASSESSMENT";
    case UUID_CHARACTERISTIC_EMAIL_ADDRESS: return "EMAIL_ADDRESS";
    case UUID_CHARACTERISTIC_FAT_BURN_HEART_RATE_LOWER_LIMIT: return "FAT_BURN_HEART_RATE_LOWER_LIMIT";
    case UUID_CHARACTERISTIC_FAT_BURN_HEART_RATE_UPPER_LIMIT: return "FAT_BURN_HEART_RATE_UPPER_LIMIT";
    case UUID_CHARACTERISTIC_FIRST_NAME: return "FIRST_NAME";
    case UUID_CHARACTERISTIC_FIVE_ZONE_HEART_RATE_LIMITS: return "FIVE_ZONE_HEART_RATE_LIMITS";
    case UUID_CHARACTERISTIC_GENDER: return "GENDER";
    case UUID_CHARACTERISTIC_HEART_RATE_MAX: return "HEART_RATE_MAX";
    case UUID_CHARACTERISTIC_HEIGHT: return "HEIGHT";
    case UUID_CHARACTERISTIC_HIP_CIRCUMFERENCE: return "HIP_CIRCUMFERENCE";
    case UUID_CHARACTERISTIC_LAST_NAME: return "LAST_NAME";
    case UUID_CHARACTERISTIC_MAXIMUM_RECOMMENDED_HEART_RATE: return "MAXIMUM_RECOMMENDED_HEART_RATE";
    case UUID_CHARACTERISTIC_RESTING_HEART_RATE: return "RESTING_HEART_RATE";
    case UUID_CHARACTERISTIC_SPORT_TYPE_FOR_AEROBIC_AND_ANAEROBIC_THRESHOLDS: return "SPORT_TYPE_FOR_AEROBIC_AND_ANAEROBIC_THRESHOLDS";
    case UUID_CHARACTERISTIC_THREE_ZONE_HEART_RATE_LIMITS: return "THREE_ZONE_HEART_RATE_LIMITS";
    case UUID_CHARACTERISTIC_TWO_ZONE_HEART_RATE_LIMITS: return "TWO_ZONE_HEART_RATE_LIMITS";
    case UUID_CHARACTERISTIC_VO2_MAX: return "VO2_MAX";
    case UUID_CHARACTERISTIC_WAIST_CIRCUMFERENCE: return "WAIST_CIRCUMFERENCE";
    case UUID_CHARACTERISTIC_WEIGHT: return "WEIGHT";
    case UUID_CHARACTERISTIC_DATABASE_CHANGE_INCREMENT: return "DATABASE_CHANGE_INCREMENT";
    case UUID_CHARACTERISTIC_USER_INDEX: return "USER_INDEX";
    case UUID_CHARACTERISTIC_BODY_COMPOSITION_FEATURE: return "BODY_COMPOSITION_FEATURE";
    case UUID_CHARACTERISTIC_BODY_COMPOSITION_MEASUREMENT: return "BODY_COMPOSITION_MEASUREMENT";
    case UUID_CHARACTERISTIC_WEIGHT_MEASUREMENT: return "WEIGHT_MEASUREMENT";
    case UUID_CHARACTERISTIC_WEIGHT_SCALE_FEATURE: return "WEIGHT_SCALE_FEATURE";
    case UUID_CHARACTERISTIC_USER_CONTROL_POINT: return "USER_CONTROL_POINT";
    case UUID_CHARACTERISTIC_MAGNETIC_FLUX_DENSITY_2D: return "MAGNETIC_FLUX_DENSITY_2D";
    case UUID_CHARACTERISTIC_MAGNETIC_FLUX_DENSITY_3D: return "MAGNETIC_FLUX_DENSITY_3D";
    case UUID_CHARACTERISTIC_LANGUAGE: return "LANGUAGE";
    case UUID_CHARACTERISTIC_BAROMETRIC_PRESSURE_TREND: return "BAROMETRIC_PRESSURE_TREND";
    case UUID_CHARACTERISTIC_BOND_MANAGEMENT_CONTROL_POINT: return "BOND_MANAGEMENT_CONTROL_POINT";
    case UUID_CHARACTERISTIC_BOND_MANAGEMENT_FEATURE: return "BOND_MANAGEMENT_FEATURE";
    case UUID_CHARACTERISTIC_CENTRAL_ADDRESS_RESOLUTION: return "CENTRAL_ADDRESS_RESOLUTION";
    case UUID_CHARACTERISTIC_CGM_MEASUREMENT: return "CGM_MEASUREMENT";
    case UUID_CHARACTERISTIC_CGM_FEATURE: return "CGM_FEATURE";
    case UUID_CHARACTERISTIC_CGM_STATUS: return "CGM_STATUS";
    case UUID_CHARACTERISTIC_CGM_SESSION_START_TIME: return "CGM_SESSION_START_TIME";
    case UUID_CHARACTERISTIC_CGM_SESSION_RUN_TIME: return "CGM_SESSION_RUN_TIME";
    case UUID_CHARACTERISTIC_CGM_SPECIFIC_OPS_CONTROL_POINT: return "CGM_SPECIFIC_OPS_CONTROL_POINT";
    case UUID_CHARACTERISTIC_INDOOR_POSITIONING_CONFIGURATION: return "INDOOR_POSITIONING_CONFIGURATION";
    case UUID_CHARACTERISTIC_LATITUDE: return "LATITUDE";
    case UUID_CHARACTERISTIC_LONGITUDE: return "LONGITUDE";
    case UUID_CHARACTERISTIC_LOCAL_NORTH_COORDINATE: return "LOCAL_NORTH_COORDINATE";
    case UUID_CHARACTERISTIC_LOCAL_EAST_COORDINATE: return "LOCAL_EAST_COORDINATE";
    case UUID_CHARACTERISTIC_FLOOR_NUMBER: return "FLOOR_NUMBER";
    case UUID_CHARACTERISTIC_ALTITUDE: return "ALTITUDE";
    case UUID_CHARACTERISTIC_UNCERTAINTY: return "UNCERTAINTY";
    case UUID_CHARACTERISTIC_LOCATION_NAME: return "LOCATION_NAME";
    case UUID_CHARACTERISTIC_URI: return "URI";
    case UUID_CHARACTERISTIC_HTTP_HEADERS: return "HTTP_HEADERS";
    case UUID_CHARACTERISTIC_HTTP_STATUS_CODE: return "HTTP_STATUS_CODE";
    case UUID_CHARACTERISTIC_HTTP_ENTITY_BODY: return "HTTP_ENTITY_BODY";
    case UUID_CHARACTERISTIC_HTTP_CONTROL_POINT: return "HTTP_CONTROL_POINT";
    case UUID_CHARACTERISTIC_HTTPS_SECURITY: return "HTTPS_SECURITY";
    default: return "";
    }
}

const char *str_ble_uuid_descriptor(uint16_t uuid)
{
    switch (uuid)
    {
    case UUID_DESCRIPTOR_CHARACTERISTIC_EXTENDED_PROPERTIES:
        return "EXTENDED_PROPERTIES";
    case UUID_DESCRIPTOR_CHARACTERISTIC_USER_DESCRIPTION:
        return "USER_DESCRIPTION";
    case UUID_DESCRIPTOR_CLIENT_CHARACTERISTIC_CONFIGURATION:
        return "CLIENT_CHARACTERISTIC_CONFIGURATION";
    case UUID_DESCRIPTOR_SERVER_CHARACTERISTIC_CONFIGURATION:
        return "SERVER_CHARACTERISTIC_CONFIGURATION";
    case UUID_DESCRIPTOR_CHARACTERISTIC_PRESENTATION_FORMAT:
        return "CHARACTERISTIC_PRESENTATION_FORMAT";
    case UUID_DESCRIPTOR_CHARACTERISTIC_AGGREGATE_FORMAT:
        return "CHARACTERISTIC_AGGREGATE_FORMAT";
    case UUID_DESCRIPTOR_VALID_RANGE:
        return "VALID_RANGE";
    case UUID_DESCRIPTOR_EXTERNAL_REPORT_REFERENCE:
        return "EXTERNAL_REPORT_REFERENCE";
    case UUID_DESCRIPTOR_REPORT_REFERENCE:
        return "REPORT_REFERENCE";
    case UUID_DESCRIPTOR_NUMBER_OF_DIGITALS:
        return "NUMBER_OF_DIGITALS";
    case UUID_DESCRIPTOR_VALUE_TRIGGER_SETTING:
        return "VALUE_TRIGGER_SETTING";
    case UUID_DESCRIPTOR_ENVIRONMENT_SENSING_CONFIGURATION:
        return "ENVIRONMENT_SENSING_CONFIGURATION";
    case UUID_DESCRIPTOR_ENVIRONMENT_SENSING_MEASUREMENT:
        return "ENVIRONMENT_SENSING_MEASUREMENT";
    case UUID_DESCRIPTOR_ENVIRONMENT_SENSING_TRIGGER_SETTING:
        return "ENVIRONMENT_SENSING_TRIGGER_SETTING";
    case UUID_DESCRIPTOR_TIME_TRIGGER_SETTING:
        return "TIME_TRIGGER_SETTING";
    default:
        return "";
    }
}

const char *str_ble_advert_type(uint8_t t)
{
    static char s[100];
    switch (t)
    {
    case BLE_ADVERT_TYPE__FLAG: return "FLAG";
    case BLE_ADVERT_TYPE__16SRV_PARTIAL: return "16SRV_PARTIAL";
    case BLE_ADVERT_TYPE__16SRV_COMPLETE: return "16SRV_COMPLETE";
    case BLE_ADVERT_TYPE__32SRV_PARTIAL: return "32SRV_PARTIAL";
    case BLE_ADVERT_TYPE__32SRV_COMPLETE: return "32SRV_COMPLETE";
    case BLE_ADVERT_TYPE__128SRV_PARTIAL: return "128SRV_PARTIAL";
    case BLE_ADVERT_TYPE__128SRV_COMPLETE: return "128SRV_COMPLETE";
    case BLE_ADVERT_TYPE__NAME_SHORT: return "NAME_SHORT";
    case BLE_ADVERT_TYPE__NAME_COMPLETE: return "NAME_COMPLETE";
    case BLE_ADVERT_TYPE__TX_POWER: return "TX_POWER";
    case BLE_ADVERT_TYPE__DEV_CLASS: return "DEV_CLASS";
    case BLE_ADVERT_TYPE__SM_TEMP_KEY: return "SM_TK";
    case BLE_ADVERT_TYPE__SM_OOB_FLAG: return "SM_OOB_FLAG";
    case BLE_ADVERT_TYPE__INTERVAL_RANGE: return "INTERVAL_RANGE";
    case BLE_ADVERT_TYPE__SOLICITATION_SRV_UUID: return "SOLICITATION_SRV_UUID";
    case BLE_ADVERT_TYPE__128SOLICITATION_SRV_UUID: return "128SOLICITATION_SRV_UUID";
    case BLE_ADVERT_TYPE__16SERVICE_DATA: return "SERVICE_DATA";
    case BLE_ADVERT_TYPE__PUBLIC_TARGET: return "PUBLIC_TARGET";
    case BLE_ADVERT_TYPE__RANDOM_TARGET: return "RANDOM_TARGET";
    case BLE_ADVERT_TYPE__APPEARANCE: return "APPEARANCE";
    case BLE_ADVERT_TYPE__ADVERT_INTERVAL: return "ADVERT_INTERVAL";
    case BLE_ADVERT_TYPE__32SOLICITATION_SRV_UUID: return "32SOLICITATION_SRV_UUID";
    case BLE_ADVERT_TYPE__32SERVICE_DATA: return "32SERVICE_DATA";
    case BLE_ADVERT_TYPE__128SERVICE_DATA: return "128SERVICE_DATA";
    case BLE_ADVERT_TYPE__MANUFACTURER_DATA: return "MANUFACTURER";
    default:
        snprintf(s, sizeof(s), "<UNKNOWN> (0x%X)", t);
        return s;
    }
}

const char *str_scan_event(uint8_t e)
{
    switch (e)
    {
    case BLE_SCAN_EVENT_TYPE__CONNECTABLE_ADVERTISEMENT:
        return "CONNECTABLE_ADVERTISEMENT";
    case BLE_SCAN_EVENT_TYPE__CONNECTABLE_DIRECTED_ADVERTISEMENT:
        return "CONNECTABLE_DIRECTED_ADVERTISEMENT";
    case BLE_SCAN_EVENT_TYPE__SCANNABLE_ADVERTISEMENT:
        return "SCANNABLE_ADVERTISEMENT";
    case BLE_SCAN_EVENT_TYPE__NON_CONNECTABLE_ADVERTISEMENT:
        return "NON_CONNECTABLE_ADVERTISEMENT";
    case BLE_SCAN_EVENT_TYPE__SCAN_RESPONSE:
        return "SCAN_RESPONSE";
    default:
        return "<UNKNOWN>";
    }
};

uint8_t scan_event_code_from_string(const char *s)
{
    if (strcasecmp(s, "CONNECTABLE_ADVERTISEMENT") == 0)
    {
        return BLE_SCAN_EVENT_TYPE__CONNECTABLE_ADVERTISEMENT;
    }
    else if (strcasecmp(s, "CONNECTABLE_DIRECTED_ADVERTISEMENT") == 0)
    {
        return BLE_SCAN_EVENT_TYPE__CONNECTABLE_DIRECTED_ADVERTISEMENT;
    }
    else if (strcasecmp(s, "SCANNABLE_ADVERTISEMENT") == 0)
    {
        return BLE_SCAN_EVENT_TYPE__SCANNABLE_ADVERTISEMENT;
    }
    else if (strcasecmp(s, "NON_CONNECTABLE_ADVERTISEMENT") == 0)
    {
        return BLE_SCAN_EVENT_TYPE__NON_CONNECTABLE_ADVERTISEMENT;
    }
    else if (strcasecmp(s, "SCAN_RESPONSE") == 0)
    {
        return BLE_SCAN_EVENT_TYPE__SCAN_RESPONSE;
    }
    return (uint8_t)-1;
}

const char *str_ble_advert_flags(uint8_t f)
{
    static char s[100];
    memset(s, 0, sizeof(s));

    if (f & BLE_ADVERT_FLAG__LIMITED_DISCOVERABLE)
    {
        strlcat(s, "LIMITED_DISCOVERABLE | ", sizeof(s));
    }
    if (f & BLE_ADVERT_FLAG__GENERAL_DISCOVERABLE)
    {
        strlcat(s, "GENERAL_DISCOVERABLE | ", sizeof(s));
    }
    if (f & BLE_ADVERT_FLAG__BREDR_NOT_SUPPORTED)
    {
        strlcat(s, "BREDR_NOT_SUPPORTED | ", sizeof(s));
    }
    if (f & BLE_ADVERT_FLAG__DUAL_MODE_SAME_CONTROLLER)
    {
        strlcat(s, "DUAL_MODE_SAME_CONTROLLER | ", sizeof(s));
    }
    if (f & BLE_ADVERT_FLAG__DUAL_MODE_SAME_HOST)
    {
        strlcat(s, "DUAL_MODE_SAME_HOST | ", sizeof(s));
    }
    while (s[strlen(s) - 1] == ' ' || s[strlen(s) - 1] == '|') s[strlen(s) - 1] = 0;
    return s;
}

const char *str_ble_uuid_attribute(uint16_t uuid)
{
    switch (uuid)
    {
    case UUID_ATTRIBUTE_PRIMARY_SERVICE:
        return "PRIMARY_SERVICE";
    case UUID_ATTRIBUTE_SECONDARY_SERVICE:
        return "SECONDARY_SERVICE";
    case UUID_ATTRIBUTE_INCLUDE:
        return "INCLUDE";
    case UUID_ATTRIBUTE_CHARACTERISTIC:
        return "CHARACTERISTIC";
    default:
        return "";
    }
}

const char *str_ble_uuid(uint16_t uuid)
{
    const char *p = str_ble_uuid_service(uuid);
    if (*p == 0)
    {
        p = str_ble_uuid_characteristic(uuid);
    }
    if (*p == 0)
    {
        p = str_ble_uuid_descriptor(uuid);
    }
    if (*p == 0)
    {
        p = str_ble_uuid_attribute(uuid);
    }
    if (*p == 0)
    {
        p = "<Unknown>";
    }
    return p;
}

const char *str_ble_gatt_appearance(uint16_t a)
{
    switch (a)
    {
    case GATT_APPEARANCE__GENERIC_PHONE:
        return "GENERIC_PHONE";
    case GATT_APPEARANCE__GENERIC_COMPUTER:
        return "GENERIC_COMPUTER";
    case GATT_APPEARANCE__GENERIC_WATCH:
        return "GENERIC_WATCH";
    case GATT_APPEARANCE__WATCH_SPORTS:
        return "SPORTS_WATCH";
    case GATT_APPEARANCE__GENERIC_CLOCK:
        return "GENERIC_CLOCK";
    case GATT_APPEARANCE__GENERIC_DISPLAY:
        return "GENERIC_DISPLAY";
    case GATT_APPEARANCE__GENERIC_REMOTE_CONTROL:
        return "GENERIC_REMOTE_CONTROL";
    case GATT_APPEARANCE__GENERIC_EYE_GLASSES:
        return "GENERIC_EYE_GLASSES";
    case GATT_APPEARANCE__GENERIC_TAG:
        return "GENERIC_TAG";
    case GATT_APPEARANCE__GENERIC_KEYRING:
        return "GENERIC_KEYRING";
    case GATT_APPEARANCE__GENERIC_MEDIA_PLAYER:
        return "GENERIC_MEDIA_PLAYER";
    case GATT_APPEARANCE__GENERIC_BARCODE_SCANNER:
        return "GENERIC_BARCODE_SCANNER";
    case GATT_APPEARANCE__GENERIC_THERMOMETER:
        return "GENERIC_THERMOMETER";
    case GATT_APPEARANCE__THERMOMETER_EAR:
        return "GENERIC_EAR_THERMOMETER";
    case GATT_APPEARANCE__GENERIC_HEART_RATE_SENSOR:
        return "GENERIC_HEART_RATE_SENSOR";
    case GATT_APPEARANCE__HEART_RATE_BELT:
        return "BELT_HEART_RATE_SENSOR";
    case GATT_APPEARANCE__GENERIC_BLOOD_PRESSURE:
        return "GENERIC_BLOOD_PRESSURE_SENSOR";
    case GATT_APPEARANCE__BLOOD_PRESSURE_ARM:
        return "ARM_BLOOD_PRESSURE_SENSOR";
    case GATT_APPEARANCE__BLOOD_PRESSURE_WRIST:
        return "WRIST_BLOOD_PRESSURE_SENSOR";
    case GATT_APPEARANCE__GENERIC_HID_DEVICE:
        return "GENERIC_HID_DEVICE";
    case GATT_APPEARANCE__HID_KEYBOARD:
        return "HID_KEYBOARD";
    case GATT_APPEARANCE__HID_MOUSE:
        return "HID_MOUSE";
    case GATT_APPEARANCE__HID_JOYSTICK:
        return "HID_JOYSTICK";
    case GATT_APPEARANCE__HID_GAMEPAD:
        return "HID_GAMEPAD";
    case GATT_APPEARANCE__HID_DIGITIZER_TABLET:
        return "HID_DIGITIZER_TABLET";
    case GATT_APPEARANCE__HID_CARD_READER:
        return "HID_CARD_READER";
    case GATT_APPEARANCE__HID_DIGITAL_PEN:
        return "HID_DIGITAL_PEN";
    case GATT_APPEARANCE__HID_BARCODE_SCANNER:
        return "HID_BARCODE_SCANNER";
    case GATT_APPEARANCE__GENERIC_GLUCOSE_METER:
        return "GENERIC_GLUCOSE_METER";
    case GATT_APPEARANCE__GENERIC_RUNNING_WALKING_SENSOR:
        return "GENERIC_RUNNING_WALKING_SENSOR";
    case GATT_APPEARANCE__RUNNING_WALKING_SENSOR_IN_SHOE:
        return "RUNNING_WALKING_SENSOR_IN_SHOE";
    case GATT_APPEARANCE__RUNNING_WALKING_SENSOR_ON_SHOE:
        return "RUNNING_WALKING_SENSOR_ON_SHOE";
    case GATT_APPEARANCE__RUNNING_WALKING_SENSOR_ON_HIP:
        return "RUNNING_WALKING_SENSOR_ON_HIP";
    case GATT_APPEARANCE__GENERIC_CYCLING:
        return "GENERIC_CYCLING";
    case GATT_APPEARANCE__CYCLING_COMPUTER:
        return "CYCLING_COMPUTER";
    case GATT_APPEARANCE__CYCLING_SPEED_SENSOR:
        return "CYCLING_SPEED_SENSOR";
    case GATT_APPEARANCE__CYCLING_CADENCE_SENSOR:
        return "CYCLING_CADENCE_SENSOR";
    case GATT_APPEARANCE__CYCLING_POWER_SENSOR:
        return "CYCLING_POWER_SENSOR";
    case GATT_APPEARANCE__CYCLING_SPEED_AND_CADENCE_SENSOR:
        return "CYCLING_SPEED_AND_CADENCE_SENSOR";
    default:
        return "<UNKNOWN>";
    }
}

const char *str_gatt_status(uint8_t gs)
{
    switch (gs)
    {
    case LTRX_GATT_STATUS__SUCCESS: return "Success";
    case LTRX_GATT_STATUS__INVALID_HANDLE: return "Invalid Handle";
    case LTRX_GATT_STATUS__READ_NOT_PERMITTED: return "Read Not Permitted";
    case LTRX_GATT_STATUS__WRITE_NOT_PERMITTED: return "Write Not permitted";
    case LTRX_GATT_STATUS__INVALID_PDU: return "Invalid PDU";
    /*
    case LTRX_GATT_STATUS__INSUF_AUTHENTICATION: return "Insufficient Authentication";
    */
    case LTRX_GATT_STATUS__REQUEST_NOT_SUPPORTED: return "Request Not Supported";
    case LTRX_GATT_STATUS__INVALID_OFFSET: return "Invalid Offset";
    /*
    case LTRX_GATT_STATUS__INSUF_AUTHORIZATION: return "Insufficient Authorization";
    */
    case LTRX_GATT_STATUS__PREPARE_QUE_FULL: return "Prepare Queue Full";
    case LTRX_GATT_STATUS__NOT_FOUND: return "Not Found";
    /*
    case LTRX_GATT_STATUS__NOT_LONG: return "Not Long Size";
    case LTRX_GATT_STATUS__INSUF_KEY_SIZE: return "Insufficient Key Size";
    */
    case LTRX_GATT_STATUS__INVALID_ATTRIBUTE_LENGTH: return "Invalid Attribute Length";
    /*
    case LTRX_GATT_STATUS__ERR_UNLIKELY: return "Error Unlikely";
    case LTRX_GATT_STATUS__INSUF_ENCRYPTION: return "Insufficient Encryption";
    case LTRX_GATT_STATUS__UNSUPPORT_GRP_TYPE: return "Unsupported Group Type";
    case LTRX_GATT_STATUS__INSUF_RESOURCE: return "Insufficient Resource";
    */
    case LTRX_GATT_STATUS__NO_RESOURCES: return "No Resources";
    case LTRX_GATT_STATUS__INTERNAL_ERROR: return "Internal Error";
    case LTRX_GATT_STATUS__WRONG_STATE: return "Wrong State";
    case LTRX_GATT_STATUS__DB_FULL: return "DB Full";
    case LTRX_GATT_STATUS__BUSY: return "Busy";
    case LTRX_GATT_STATUS__ERROR: return "Error";
    case LTRX_GATT_STATUS__CMD_STARTED: return "Command Started";
    case LTRX_GATT_STATUS__ILLEGAL_PARAMETER: return "Illegal Parameter";
    case LTRX_GATT_STATUS__PENDING: return "Pending";
    /*
    case LTRX_GATT_STATUS__AUTH_FAIL: return "Authentication Fail";
    case LTRX_GATT_STATUS__MORE: return "More";
    case LTRX_GATT_STATUS__INVALID_CFG: return "Invalid Configuration";
    case LTRX_GATT_STATUS__SERVICE_STARTED: return "Service Started";
    case LTRX_GATT_STATUS__ENCRYPED_NO_MITM: return "Encrypted No MITM";
    case LTRX_GATT_STATUS__NOT_ENCRYPTED: return "Not Encrypted";
    case LTRX_GATT_STATUS__CONGESTED: return "Congested";
    case LTRX_GATT_STATUS__CCC_CFG_ERR: return "Improper Client Char Configuration";
    case LTRX_GATT_STATUS__PRC_IN_PROGRESS: return "Procedure Already in Progress";
    case LTRX_GATT_STATUS__OUT_OF_RANGE: return "Value Out of Range";
    */
    default: return "Unknown";
    }
}










