/*****************************************************************************/
/*                              Legal                                        */
/*****************************************************************************/

/*
** Copyright ©2020-2025, Lantronix, Inc. All Rights Reserved.
** By using this software, you are agreeing to the terms of the Software
** Development Kit (SDK) License Agreement included in the distribution package
** for this software (the “License Agreement”).
** Under the License Agreement, this software may be used solely to create
** custom applications for use on the Lantronix xPico Wi-Fi product.
** THIS SOFTWARE AND ANY ACCOMPANYING DOCUMENTATION IS PROVIDED "AS IS".
** LANTRONIX SPECIFICALLY DISCLAIMS ANY WARRANTIES, INCLUDING, BUT NOT LIMITED
** TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY, NON-INFRINGEMENT AND FITNESS
** FOR A PARTICULAR PURPOSE.
** LANTRONIX HAS NO OBLIGATION TO PROVIDE MAINTENANCE, SUPPORT, UPDATES,
** ENHANCEMENTS, OR MODIFICATIONS TO THIS SOFTWARE.
** IN NO EVENT SHALL LANTRONIX BE LIABLE TO ANY PARTY FOR DIRECT, INDIRECT,
** SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES, INCLUDING LOST PROFITS,
** ARISING OUT OF THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF
** LANTRONIX HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
*/

/****************************************************************************/
/*                                 Includes                                 */
/****************************************************************************/

#include <inttypes.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include "ble_explorer_module_defs.h"
#include "ble_strings.h"
#include "ble_utils.h"
#include "bluetooth_module_libs.h"
#include "ctype.h"
#include "ltrx_cfgvar.h"
#include "ltrx_definitions.h"
#include "ltrx_bluetooth.h"
#include "ltrx_gatt.h"
#include "ltrx_stream.h"
#include "ltrx_tlog.h"
#include "ltrx_xml.h"
#include "uuid.h"


/****************************************************************************/
/*                                  Macros                                  */
/****************************************************************************/

#define BLE_ADVERT_WAIT_DEFAULT  5

#define BLE_ADVERT_DISPLAY_MODE__BINARY     0x01

#define BLE_ADVERT_DISPLAY_MODE__FORMATTED  0x02


/****************************************************************************/
/*                              Data Structures                             */
/****************************************************************************/

struct ble_advert_callback_params
{
    uint32_t count;
    uint8_t mode;
    struct output_stream *os;
};


/****************************************************************************/
/*                                  Locals                                  */
/****************************************************************************/


/****************************************************************************/
/*                           Function Definitions                           */
/****************************************************************************/

static void ble_advert_callback(
    const struct ltrx_ble_advert *lba, void *arg
)
{
    struct ble_advert_callback_params *bacp = arg;
    bacp->count++;

    if (bacp->mode & BLE_ADVERT_DISPLAY_MODE__BINARY)
    {
#if 1
        uint32_t max_length = (
            sizeof(*lba) - sizeof(lba->lba_data) + (lba->lba_data_count * sizeof(lba->lba_data[0]))
        );
        uint8_t *p = MALLOC_ZEROED(max_length);
        uint32_t length = 0;
        if (p)
        {
            uint8_t *q = p;
            length = sizeof(*lba) - sizeof(lba->lba_data);
            memcpy(p, lba, length);
            q += length;
            TLOG(7, "length start: %u  count: %u", length, lba->lba_data_count);
            for (uint8_t i = 0; i < lba->lba_data_count; ++i)
            {
                uint32_t data_length = (
                    sizeof(lba->lba_data[i]) 
                    - sizeof(lba->lba_data[i].lbad_content)
                    + (sizeof(uint8_t) * lba->lba_data[i].lbad_length)
                );
                memcpy(q, &lba->lba_data[i], data_length);
                q += data_length;
                length += data_length;
                TLOG(7, "  %u lbad_len: %u  data_len: %u  total: %u",
                    i, lba->lba_data[i].lbad_length, data_length, length);
            }
        }
#else
        uint32_t length = (
            sizeof(*lba) - sizeof(lba->lba_data) + (lba->lba_data_count * sizeof(lba->lba_data[0]))
        );
        const uint8_t *p = (const uint8_t *)lba;
#endif
        if (p && length)
        {
            ltrx_output_stream_hexdump(bacp->os, p, length, 0);
            stream_printf(bacp->os, "\r\n");
        }
    }

    if (bacp->mode & BLE_ADVERT_DISPLAY_MODE__FORMATTED)
    {
        stream_printf(
            bacp->os, "%M  %s  RSSI: %d\r\n",
            lba->lba_bda,
            lba->lba_address_type < 4 /* ARRAY_SIZE(g_ble_addr_type_desc) */
                ? g_ble_addr_type_desc[lba->lba_address_type]
                : "unknown",
            lba->lba_rssi
        );
        stream_printf(
            bacp->os, "  Event: %s (0x%X)\r\n",
            str_scan_event(lba->lba_scan_event_type),
            lba->lba_scan_event_type
        );
        stream_printf(
            bacp->os, "  Flags: %s (0x%X)\r\n",
            str_ble_advert_flags(lba->lba_advert_flags),
            lba->lba_advert_flags
        );
        stream_printf(
            bacp->os, "  %u data element%s%s\r\n",
            lba->lba_data_count,
            lba->lba_data_count == 1 ? "" : "s",
            lba->lba_data_count ? ":" : ""
        );

        for (
            uint8_t i = 0;
            i < MINIMUM(lba->lba_data_count, LTRX_BLE_ADVERT_DATA_MAX_ELEMENTS);
            ++i
        )
        {
            stream_printf(
                bacp->os, "    %u: (0x%X) %s",
                i, lba->lba_data[i].lbad_type, str_ble_advert_type(lba->lba_data[i].lbad_type)
            );
            if (lba->lba_data[i].lbad_length)
            {
                if (
                    lba->lba_data[i].lbad_type == BLE_ADVERT_TYPE__NAME_SHORT ||
                    lba->lba_data[i].lbad_type == BLE_ADVERT_TYPE__NAME_COMPLETE
                )
                {
                    stream_printf(
                        bacp->os, " [%u]: '%*.*s'\r\n",
                        lba->lba_data[i].lbad_length,
                        lba->lba_data[i].lbad_length,
                        lba->lba_data[i].lbad_length,
                        lba->lba_data[i].lbad_content
                    );
                }
                else if (
                    lba->lba_data[i].lbad_type == BLE_ADVERT_TYPE__TX_POWER
                )
                {
                    int8_t p = lba->lba_data[i].lbad_content[0];
                    stream_printf(bacp->os, ": %d\r\n", p);
                }
                else
                {
                    stream_printf(
                        bacp->os, " [%u]: % *b\r\n",
                        lba->lba_data[i].lbad_length,
                        MINIMUM(
                            lba->lba_data[i].lbad_length,
                            BLE_ADVERT_DATA_MAX_LENGTH
                        ),
                        lba->lba_data[i].lbad_content
                    );
                }
            }
            else
            {
                stream_printf(bacp->os, "\r\n");
            }
        }
    }
    ltrx_output_stream_write_line(
        bacp->os,
        "------------------------------------------------------"
        "----------------------------------------------------"
    );
    ltrx_output_stream_flush_data(bacp->os);
}

bool BLEWatchAds(
    uint32_t zeroBasedIndex,
    const char *value,
    const struct ltrx_write_user_message_info *lwumi
)
{
    (void)zeroBasedIndex;
    VERIFY_CLI_ENVIRONMENT_OR_ABORT(lwumi);

    struct output_stream *os = ltrx_cli_get_output_stream(
        lwumi->optCliThreadInfo
    );
    struct input_stream *is = ltrx_cli_get_input_stream(
        lwumi->optCliThreadInfo
    );
    static const char options[] = "a:d:rRs:";
    struct buffer_arg_context bac;
    char *optarg = NULL;
    int c;
    char *arg;
    uint32_t duration = BLE_ADVERT_WAIT_DEFAULT;
    struct ble_advert_callback_params bacp = {
        .count = 0,
        .mode = BLE_ADVERT_DISPLAY_MODE__FORMATTED,
        .os = os,
    };
    struct ltrx_ble_advert_filter lbaf;
    memset(&lbaf, 0, sizeof(lbaf));

    if (! buffer_arg_init(value, &bac, NULL))
    {
        return false;
    }

    while ((c = buffer_arg_getopt(&bac, options, &optarg)) != -1)
    {
        switch (c)
        {
        case 'a':
            arg = optarg;
            if (strcasecmp(arg, "-e") == 0)
            {
                lbaf.device.lbdf_filter_type = LTRX_BLE_ADVERT_FILTER_TYPE__EXCLUDE;
                arg = buffer_arg_next(&bac);
            }
            else
            {
                lbaf.device.lbdf_filter_type = LTRX_BLE_ADVERT_FILTER_TYPE__INCLUDE;
            }
            {
                char *last;
                char *p = strtok_r(arg, ";", &last);
                if (p && *p != '-')
                {
                    do {
                        uint8_t bda[BD_ADDRESS_SIZE];
                        if (str_bdaddr(p, bda))
                        {
                            if (lbaf.device.lbdf_count < ARRAY_SIZE(lbaf.device.lbdf_addresses))
                            {
                                memcpy(
                                    &lbaf.device.lbdf_addresses[lbaf.device.lbdf_count],
                                    bda,
                                    sizeof(lbaf.device.lbdf_addresses[lbaf.device.lbdf_count])
                                );
                                lbaf.device.lbdf_count++;
                            }
                            else
                            {
                                stream_printf(
                                    os, "Ignoring excess device address '%M'.\r\n",
                                    lbaf.device.lbdf_addresses[lbaf.device.lbdf_count]
                                );
                            }
                        }
                        else
                        {
                            stream_printf(
                                os, "Ignoring invalid device address '%M'.\r\n",
                                lbaf.device.lbdf_addresses[lbaf.device.lbdf_count]
                            );
                        }
                        p = strtok_r(NULL, ";", &last);

                    } while (p);
                }
            }
            break;

        case 'd':
            duration = strtoul(optarg, NULL, 0);
            break;

        case 'r':
            bacp.mode |= BLE_ADVERT_DISPLAY_MODE__BINARY;
            break;

        case 'R':
            bacp.mode |= BLE_ADVERT_DISPLAY_MODE__BINARY;
            bacp.mode &= ~BLE_ADVERT_DISPLAY_MODE__FORMATTED;
            break;

        case 's':
            arg = optarg;
            if (strcasecmp(arg, "-e") == 0)
            {
                lbaf.scan_event.lbef_filter_type = LTRX_BLE_ADVERT_FILTER_TYPE__EXCLUDE;
                arg = buffer_arg_next(&bac);
            }
            else
            {
                lbaf.scan_event.lbef_filter_type = LTRX_BLE_ADVERT_FILTER_TYPE__INCLUDE;
            }
            {
                char *last;
                char *p = strtok_r(arg, ";", &last);
                if (p && *p != '-')
                {
                    do {
                        bool numeric_code = true;
                        for (uint8_t i = 0; p[i]; ++i)
                        {
                            if (! isdigit(p[i]))
                            {
                                numeric_code = false;
                                break;
                            }
                        }
                        if (numeric_code)
                        {
                            if (
                                lbaf.scan_event.lbef_count
                                    < ARRAY_SIZE(lbaf.scan_event.lbef_scan_events)
                            )
                            {
                                lbaf.scan_event.lbef_scan_events[
                                    lbaf.scan_event.lbef_count
                                ] = strtoul(p, NULL, 0);
                                lbaf.scan_event.lbef_count++;
                            }
                            else
                            {
                                stream_printf(
                                    os, "Ignoring excess scan event '%s'.\r\n", p
                                );
                            }
                        }
                        else
                        {
                            uint8_t e = scan_event_code_from_string(p);
                            if (e == (uint8_t)-1)
                            {
                                stream_printf(
                                    os, "Ignoring invalid scan event '%s'.\r\n", p
                                );
                                ltrx_thread_sleep(1000);
                            }
                            else
                            {
                                lbaf.scan_event.lbef_scan_events[
                                    lbaf.scan_event.lbef_count
                                ] = e;
                                lbaf.scan_event.lbef_count++;
                            }
                        }
                        p = strtok_r(NULL, ";", &last);

                    } while (p);
                }
            }
            break;

        default:
            break;
        }
    }
    buffer_arg_term(&bac);

    if (duration == 0)
    {
        duration = 1;
    }
    ltrx_output_stream_write_line(
        os, "\r\nWorking. Hit any key to abort.\r\n"
    );
    ltrx_output_stream_flush_data(os);

    if (
        ! ltrx_ble_advert_notification_register(
            lbaf.device.lbdf_count || lbaf.scan_event.lbef_count ? &lbaf : NULL,
            ble_advert_callback,
            &bacp
        )
    )
    {
        ltrx_output_stream_write_line(
            os, "ERROR: Failed to register notification callback."
        );
    }
    else
    {
        uint32_t st = ltrx_timemark();
        for (uint32_t i = 0; i < duration * 1000; ++i)
        {
            if (ltrx_input_stream_peek_with_block_time(is, 1) >= 0)
            {
                ltrx_input_stream_read(is);
                ltrx_output_stream_write_line(os, "\r\n<ABORT>");
                break;
            }
        }
        ltrx_ble_advert_notification_unregister(
            ble_advert_callback
        );
        stream_printf(
            os,
            "\r\nDuration: %u seconds  Count: %u\r\n",
            ltrx_elapsed_time_current_ms(st) / 1000,
            bacp.count
        );
    }
    return true;
}

bool BLEShowDevices(
    uint32_t zeroBasedIndex,
    const struct ltrx_write_user_message_info *lwumi
)
{
    (void)zeroBasedIndex;
    VERIFY_CLI_ENVIRONMENT_OR_ABORT(lwumi);

    const struct StatusInfo *si = &g_bluetoothModule->status_table[
        STATUSDEF_ELEMENT_GROUP_BLUETOOTH_NEARBY_DEVICES
    ];
    if (lwumi->optStatusContainedShow)
    {
        lwumi->optStatusContainedShow(0, si, lwumi);
    }
    return true;
}

bool BLEScan(
    uint32_t zeroBasedIndex,
    const struct ltrx_write_user_message_info *lwumi
)
{
    (void)zeroBasedIndex;
    VERIFY_CLI_ENVIRONMENT_OR_ABORT(lwumi);

    const struct StatusInfo *si = &g_bluetoothModule->status_table[
        STATUSDEF_ELEMENT_ITEM_BLUETOOTH_NEARBY_DEVICES_SCAN
    ];
    bool rc = si->info.action.actionFunction(0, NULL);
    if (rc)
    {
        ltrx_write_user_message(
            lwumi,
            LTRX_USER_MESSAGE_SEVERITY__INFORMATIONAL,
            "Scan started."
        );
    }
    else
    {
        ltrx_write_user_message(
            lwumi,
            LTRX_USER_MESSAGE_SEVERITY__ERROR,
            "Failed to start scan."
        );
    }
    return rc;
}



