/*****************************************************************************/
/*                              Legal                                        */
/*****************************************************************************/

/*
** Copyright ©2020-2025, Lantronix, Inc. All Rights Reserved.
** By using this software, you are agreeing to the terms of the Software
** Development Kit (SDK) License Agreement included in the distribution package
** for this software (the “License Agreement”).
** Under the License Agreement, this software may be used solely to create
** custom applications for use on the Lantronix xPico Wi-Fi product.
** THIS SOFTWARE AND ANY ACCOMPANYING DOCUMENTATION IS PROVIDED "AS IS".
** LANTRONIX SPECIFICALLY DISCLAIMS ANY WARRANTIES, INCLUDING, BUT NOT LIMITED
** TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY, NON-INFRINGEMENT AND FITNESS
** FOR A PARTICULAR PURPOSE.
** LANTRONIX HAS NO OBLIGATION TO PROVIDE MAINTENANCE, SUPPORT, UPDATES,
** ENHANCEMENTS, OR MODIFICATIONS TO THIS SOFTWARE.
** IN NO EVENT SHALL LANTRONIX BE LIABLE TO ANY PARTY FOR DIRECT, INDIRECT,
** SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES, INCLUDING LOST PROFITS,
** ARISING OUT OF THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF
** LANTRONIX HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
*/

/****************************************************************************/
/*                                 Includes                                 */
/****************************************************************************/

#include <inttypes.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include "ble_explorer_module_defs.h"
#include "ble_strings.h"
#include "ble_utils.h"
#include "bluetooth_module_libs.h"
#include "ltrx_cfgvar.h"
#include "ltrx_definitions.h"
#include "ltrx_bluetooth.h"
#include "ltrx_gatt.h"
#include "ltrx_stream.h"
#include "ltrx_tlog.h"
#include "ltrx_xml.h"
#include "uuid.h"

#pragma GCC diagnostic ignored "-Wunused-function"
#pragma GCC diagnostic ignored "-Wunused-parameter"
#pragma GCC diagnostic ignored "-Wunused-variable"


/****************************************************************************/
/*                                  Macros                                  */
/****************************************************************************/

#define DISCOVERY_TIMEOUT_MS   (180 * 1000)


/****************************************************************************/
/*                             Data Structures                              */
/****************************************************************************/


/****************************************************************************/
/*                           Function Definitions                           */
/****************************************************************************/

static const char *str_gatt_properties(uint8_t p)
{
    static char s[100];
    memset(s, 0, sizeof(s));

    if (p & GATT_DB_PROPERTY__BROADCAST)
    {
        strlcat(s, "BROADCAST | ", sizeof(s));
    }
    if (p & GATT_DB_PROPERTY__READ)
    {
        strlcat(s, "READ | ", sizeof(s));
    }
    if (p & GATT_DB_PROPERTY__WRITE_NO_RESPONSE)
    {
        strlcat(s, "WRITE_NR | ", sizeof(s));
    }
    if (p & GATT_DB_PROPERTY__WRITE)
    {
        strlcat(s, "WRITE | ", sizeof(s));
    }
    if (p & GATT_DB_PROPERTY__NOTIFY)
    {
        strlcat(s, "NOTIFY | ", sizeof(s));
    }
    if (p & GATT_DB_PROPERTY__INDICATE)
    {
        strlcat(s, "INDICATE | ", sizeof(s));
    }
    if (p & GATT_DB_PROPERTY__AUTHD_WRITES)
    {
        strlcat(s, "AUTH | ", sizeof(s));
    }
    if (p & GATT_DB_PROPERTY__EXTENDED)
    {
        strlcat(s, "EXT_PROP | ", sizeof(s));
    }
    while (s[strlen(s) - 1] == ' ' || s[strlen(s) - 1] == '|')
    {
        s[strlen(s) - 1] = 0;
    }
    return s;
}

bool GATTServices(
    uint32_t zeroBasedIndex,
    const char *value,
    const struct ltrx_write_user_message_info *lwumi
)
{
    (void)zeroBasedIndex;
    VERIFY_CLI_ENVIRONMENT_OR_ABORT(lwumi);

    struct output_stream *os = ltrx_cli_get_output_stream(
        lwumi->optCliThreadInfo
    );
    struct buffer_arg_context bac;
    char *optarg = NULL;
    static const char options[] = "u:";
    int c;
    uint32_t argc;
    bool uuid = false;
    struct ltrx_uuid lu;

    if (! buffer_arg_init(value, &bac, &argc))
    {
        return false;
    }
    if (argc < 1)
    {
        ltrx_write_user_message(
            lwumi,
            LTRX_USER_MESSAGE_SEVERITY__ERROR,
            "Must specify connection id."
        );
        buffer_arg_term(&bac);
        return false;
    }
    while ((c = buffer_arg_getopt(&bac, options, &optarg)) != -1)
    {
        switch (c)
        {
        case 'u':
        case 'U':
            uuid = true;
            bt_uuid_parse(optarg, &lu);
            break;

        default:
            break;
        }
    }
    uint16_t cid = strtoul(buffer_arg_next(&bac), NULL, 0);
    buffer_arg_term(&bac);
    ltrx_write_user_message(
        lwumi,
        LTRX_USER_MESSAGE_SEVERITY__EPHEMERAL,
        "Please wait ..."
    );
    ltrx_thread_sleep(2);

    struct ltrx_gatt_disc_service *lgs;
    uint16_t nlgs;

    if (
        ! ltrx_gatt_discover_services(
            cid, uuid ? &lu : NULL, &lgs, &nlgs, DISCOVERY_TIMEOUT_MS
        )
    )
    {
        ltrx_write_user_message(
            lwumi,
            LTRX_USER_MESSAGE_SEVERITY__ERROR,
            "Failed to start discovery."
        );
        return false;
    }
    stream_printf(
        os, "%u services discovered:\r\n\r\n", nlgs
    );
    if (nlgs)
    {
        stream_printf(
            os,
            "%-37s  %-28s  %s\r\n",
            "UUID", "Service Name", "Handle Range"
        );
        ltrx_output_stream_write_line(
            os,
            "-------------------------------------  "
            "----------------------------  "
            "----------------"
        );
        for (uint16_t i = 0; i < nlgs; ++i)
        {
            stream_printf(
                os,
                "%-37s  %-28.28s  0x%04X - 0x%04X\r\n",
                bt_uuid_unparse(&lgs[i].uuid, NULL, 0),
                lgs[i].uuid.length == 2
                    ? str_ble_uuid_service(lgs[i].uuid.uu.id16) : "<Proprietary>",
                lgs[i].handle_range.h_start, lgs[i].handle_range.h_end
            );
        }
        ltrx_free(lgs);
    }
    return true;
}

bool GATTCharacteristics(
    uint32_t zeroBasedIndex,
    const char *value,
    const struct ltrx_write_user_message_info *lwumi
)
{
    (void)zeroBasedIndex;
    VERIFY_CLI_ENVIRONMENT_OR_ABORT(lwumi);

    struct output_stream *os = ltrx_cli_get_output_stream(
        lwumi->optCliThreadInfo
    );
    struct buffer_arg_context bac;
    char *optarg = NULL;
    static const char options[] = "e:s:";
    int c;
    struct ltrx_gatt_handle_range hr = { .h_start = 1, .h_end = 0xFFFF, };
    uint32_t argc;

    if (! buffer_arg_init(value, &bac, &argc))
    {
        return false;
    }
    if (argc < 1)
    {
        ltrx_write_user_message(
            lwumi,
            LTRX_USER_MESSAGE_SEVERITY__ERROR,
            "Must specify connection id."
        );
        buffer_arg_term(&bac);
        return false;
    }
    while ((c = buffer_arg_getopt(&bac, options, &optarg)) != -1)
    {
        switch (c)
        {
        case 'e':
        case 'E':
            hr.h_end = (uint16_t)strtoul(optarg, NULL, 0);
            break;

        case 's':
        case 'S':
            hr.h_start = (uint16_t)strtoul(optarg, NULL, 0);
            break;

        default:
            break;
        }
    }
    uint16_t cid = strtoul(buffer_arg_next(&bac), NULL, 0);
    buffer_arg_term(&bac);
    ltrx_write_user_message(
        lwumi,
        LTRX_USER_MESSAGE_SEVERITY__EPHEMERAL,
        "Please wait ..."
    );
    ltrx_thread_sleep(2);

    struct ltrx_gatt_disc_characteristic *lgdc;
    uint16_t nlgdc;

    if (
        ! ltrx_gatt_discover_characteristics(
            cid, &hr, &lgdc, &nlgdc, DISCOVERY_TIMEOUT_MS
        )
    )
    {
        ltrx_write_user_message(
            lwumi,
            LTRX_USER_MESSAGE_SEVERITY__ERROR,
            "Failed to start discovery."
        );
        return false;
    }

    stream_printf(
        os,
        "%u characteristics discovered for handle range "
        "0x%04X - 0x%04X:\r\n\r\n",
        nlgdc, hr.h_start, hr.h_end
    );
    if (nlgdc)
    {
        stream_printf(
            os,
            "%-37s  %-28s  %-6s  %-10s  %s\r\n",
            "UUID", "Name", "Handle", "Val Handle", "Properties"
        );
        ltrx_output_stream_write_line(
            os,
            "-------------------------------------  "
            "----------------------------  "
            "------  ----------  "
            "--------------------------------"
        );
        for (uint16_t i = 0; i < nlgdc; ++i)
        {
            const char *s = "";
            if (lgdc[i].uuid.length == 2)
            {
                s = str_ble_uuid_characteristic(lgdc[i].uuid.uu.id16);
            }
            if (! *s)
            {
                s = "<Proprietary>";
            }
            stream_printf(
                os,
                "%-37s  %-28.28s  0x%04X  0x%04X      %s\r\n",
                bt_uuid_unparse(&lgdc[i].uuid, NULL, 0),
                s,
                lgdc[i].handle,
                lgdc[i].value_handle,
                str_gatt_properties(lgdc[i].properties)
            );
        }
        ltrx_free(lgdc);
    }
    return true;
}

bool GATTDescriptors(
    uint32_t zeroBasedIndex,
    const char *value,
    const struct ltrx_write_user_message_info *lwumi
)
{
    (void)zeroBasedIndex;
    VERIFY_CLI_ENVIRONMENT_OR_ABORT(lwumi);

    struct output_stream *os = ltrx_cli_get_output_stream(
        lwumi->optCliThreadInfo
    );
    struct buffer_arg_context bac;
    char *optarg = NULL;
    static const char options[] = "e:s:";
    int c;
    struct ltrx_gatt_handle_range hr = { 1, 0xFFFF };
    uint32_t argc;

    if (! buffer_arg_init(value, &bac, &argc))
    {
        return false;
    }
    if (argc < 1)
    {
        ltrx_write_user_message(
            lwumi,
            LTRX_USER_MESSAGE_SEVERITY__ERROR,
            "Must specify connection id."
        );
        buffer_arg_term(&bac);
        return false;
    }
    while ((c = buffer_arg_getopt(&bac, options, &optarg)) != -1)
    {
        switch (c)
        {
        case 'e':
        case 'E':
            hr.h_end = (uint16_t)strtoul(optarg, NULL, 0);
            break;

        case 's':
        case 'S':
            hr.h_start = (uint16_t)strtoul(optarg, NULL, 0);
            break;

        default:
            break;
        }
    }
    uint16_t cid = strtoul(buffer_arg_next(&bac), NULL, 0);
    buffer_arg_term(&bac);
    ltrx_write_user_message(
        lwumi,
        LTRX_USER_MESSAGE_SEVERITY__EPHEMERAL,
        "Please wait ..."
    );
    ltrx_thread_sleep(2);

    struct ltrx_gatt_disc_descriptor *lgdd;
    uint16_t nlgdd;

    if (
        ! ltrx_gatt_discover_characteristic_descriptors(
            cid, &hr, &lgdd, &nlgdd, DISCOVERY_TIMEOUT_MS
        )
    )
    {
        ltrx_write_user_message(
            lwumi,
            LTRX_USER_MESSAGE_SEVERITY__ERROR,
            "Failed to start discovery."
        );
        return false;
    }
    stream_printf(
        os,
        "%u characteristics discovered for handle range "
        "0x%04X - 0x%04X:\r\n\r\n",
        nlgdd, hr.h_start, hr.h_end
    );
    if (nlgdd)
    {
        stream_printf(
            os, "%-37s  %-28s  %-6s\r\n", "UUID", "Name", "Handle"
        );
        ltrx_output_stream_write_line(
            os,
            "-------------------------------------  "
            "----------------------------  ------"
        );
        for (uint16_t i = 0; i < nlgdd; ++i)
        {
            const char *s = "";
            if (lgdd[i].uuid.length == 2)
            {
                s = str_ble_uuid(lgdd[i].uuid.uu.id16);
            }
            if (! *s)
            {
                s = "<Proprietary>";
            }
            stream_printf(
                os,
                "%-37s  %-28.28s  0x%04X\r\n",
                bt_uuid_unparse(&lgdd[i].uuid, NULL, 0), s, lgdd[i].handle
            );
        }
        ltrx_free(lgdd);
    }
    return true;
}

