/*****************************************************************************/
/*                              Legal                                        */
/*****************************************************************************/

/*
** Copyright ©2020-2025, Lantronix, Inc. All Rights Reserved.
** By using this software, you are agreeing to the terms of the Software
** Development Kit (SDK) License Agreement included in the distribution package
** for this software (the “License Agreement”).
** Under the License Agreement, this software may be used solely to create
** custom applications for use on the Lantronix xPico Wi-Fi product.
** THIS SOFTWARE AND ANY ACCOMPANYING DOCUMENTATION IS PROVIDED "AS IS".
** LANTRONIX SPECIFICALLY DISCLAIMS ANY WARRANTIES, INCLUDING, BUT NOT LIMITED
** TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY, NON-INFRINGEMENT AND FITNESS
** FOR A PARTICULAR PURPOSE.
** LANTRONIX HAS NO OBLIGATION TO PROVIDE MAINTENANCE, SUPPORT, UPDATES,
** ENHANCEMENTS, OR MODIFICATIONS TO THIS SOFTWARE.
** IN NO EVENT SHALL LANTRONIX BE LIABLE TO ANY PARTY FOR DIRECT, INDIRECT,
** SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES, INCLUDING LOST PROFITS,
** ARISING OUT OF THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF
** LANTRONIX HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
*/

/****************************************************************************/
/*                                 Includes                                 */
/****************************************************************************/

#include <inttypes.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include "ble_explorer_module_defs.h"
#include "ble_strings.h"
#include "ble_utils.h"
#include "bluetooth_module_libs.h"
#include "ltrx_cfgvar.h"
#include "ltrx_definitions.h"
#include "ltrx_bluetooth.h"
#include "ltrx_gatt.h"
#include "ltrx_stream.h"
#include "ltrx_time.h"
#include "ltrx_tlog.h"
#include "ltrx_xml.h"
#include "uuid.h"

#include "../ble_test_server/bts_gatt.h"

#pragma GCC diagnostic ignored "-Wunused-function"
#pragma GCC diagnostic ignored "-Wunused-parameter"
#pragma GCC diagnostic ignored "-Wunused-variable"
#pragma GCC diagnostic ignored "-Wunused-but-set-variable"


/****************************************************************************/
/*                                  Macros                                  */
/****************************************************************************/

#define CONNECT__INCOMPLETE     0

#define CONNECT__COMPLETE       1

#define CONNECT__ABORT          2

#define BUFFER_SIZE__MIN        8

#define BUFFER_SIZE__DEFAULT    64

#define BUFFER_SIZE__MAX        4096

#define WRITE_TYPE__NORMAL      LTRX_GATT_WRITE__NORMAL

#define WRITE_TYPE__NO_RESPONSE LTRX_GATT_WRITE__NO_RESPONSE

#define WRITE_TYPE__PREPARE     LTRX_GATT_WRITE__PREPARE

#define WRITE_TYPE__EXECUTE     (LTRX_GATT_WRITE__PREPARE + 1)

#define WRITE_TYPE__CANCEL      (LTRX_GATT_WRITE__PREPARE + 2)


/****************************************************************************/
/*                             Data Structures                              */
/****************************************************************************/

struct gatt_connect_context
{
    uint8_t complete;
    bool connected;
    uint16_t connection_id;
};


/****************************************************************************/
/*                                Locals                                    */
/****************************************************************************/

static bool s_notification_listen;

static struct output_stream *s_notification_ostream;


/****************************************************************************/
/*                           Function Definitions                           */
/****************************************************************************/

bool GATTShow(
    uint32_t zeroBasedIndex,
    const struct ltrx_write_user_message_info *lwumi
)
{
    (void)zeroBasedIndex;
    VERIFY_CLI_ENVIRONMENT_OR_ABORT(lwumi);

    const struct ltrx_module_info *lmi = ltrx_module_lookup(
        "bluetooth"
    );
    const struct StatusInfo *s = &lmi->status_table[
        STATUSDEF_ELEMENT_GROUP_BLUETOOTH_OPERATION__CONTENTS_GATT_CONNECTION
    ];
    if (lwumi->optStatusContainedShow)
    {
        for (uint8_t i = 0; i < NUM_GATT_CLIENT_CONNECTIONS; ++i)
        {
            lwumi->optStatusContainedShow(i, s, lwumi);
        }
    }
    return true;
}

static void gatt_connect_callback(
    const uint8_t bda[], bool connected, uint16_t connection_id, void *arg
)
{
    struct gatt_connect_context *gcc = arg;
    gcc->connection_id = connection_id;
    gcc->connected = connected;
    gcc->complete = CONNECT__COMPLETE;
}

bool GATTConnect(
    uint32_t zeroBasedIndex,
    const char *value,
    const struct ltrx_write_user_message_info *lwumi
)
{
    (void)zeroBasedIndex;
    VERIFY_CLI_ENVIRONMENT_OR_ABORT(lwumi);

    struct output_stream *os = ltrx_cli_get_output_stream(
        lwumi->optCliThreadInfo
    );
    struct input_stream *is = ltrx_cli_get_input_stream(
        lwumi->optCliThreadInfo
    );
    struct buffer_arg_context bac;
    uint32_t argc;
    char *optarg = NULL;
    uint8_t bda[BD_ADDRESS_SIZE];
    uint8_t type = 0xFF;
    uint8_t transport = BT_TRANSPORT__LE;
    static const char options[] = "a:t:";
    int c;

    if (! buffer_arg_init(value, &bac, &argc))
    {
        return false;
    }
    if (argc < 1)
    {
        ltrx_write_user_message(
            lwumi,
            LTRX_USER_MESSAGE_SEVERITY__ERROR,
            "Must specify connection id."
        );
        buffer_arg_term(&bac);
        return false;
    }
    while ((c = buffer_arg_getopt(&bac, options, &optarg)) != -1)
    {
        switch (c)
        {
        case 'a':
            if (strcasecmp(optarg, "publicid") == 0)
            {
                type = BLE_ADDRESS_TYPE__PUBLIC_ID;
            }
            else if (strcasecmp(optarg, "randomid") == 0)
            {
                type = BLE_ADDRESS_TYPE__RANDOM_ID;
            }
            else if (strncasecmp(optarg, "random", 3) == 0)
            {
                type = BLE_ADDRESS_TYPE__RANDOM;
            }
            else if (strncasecmp(optarg, "public", 3) == 0)
            {
                type = BLE_ADDRESS_TYPE__PUBLIC;
            }
            else
            {
                ltrx_write_user_message(
                    lwumi,
                    LTRX_USER_MESSAGE_SEVERITY__ERROR,
                    "Invalid address type."
                );
            }
            break;

        case 't':
            if (strcasecmp(optarg, "bredr") == 0 || strcasecmp(optarg, "br/edr") == 0)
            {
                transport = BT_TRANSPORT__BREDR;
            }
            else if (strcasecmp(optarg, "ble") == 0 || strcasecmp(optarg, "le") == 0)
            {
                transport = BT_TRANSPORT__LE;
            }
            else
            {
                ltrx_write_user_message(
                    lwumi,
                    LTRX_USER_MESSAGE_SEVERITY__ERROR,
                    "Invalid transport."
                );
            }
            break;

        default:
            break;
        }
    }

    optarg = buffer_arg_next(&bac);
    if (! str_bdaddr(optarg, bda))
    {
        ltrx_write_user_message(
            lwumi,
            LTRX_USER_MESSAGE_SEVERITY__ERROR,
            "Invalid device address."
        );
        buffer_arg_term(&bac);
        return false;
    }
    buffer_arg_term(&bac);

    if (type == 0xFF)
    {
        struct bluetooth_device bd;
        if (ltrx_bluetooth_get_cached_peer(bda, &bd))
        {
            type = bd.bd_address_type;
        }
        else
        {
            type = BLE_ADDRESS_TYPE__PUBLIC;
        }
    }

    ltrx_write_user_message(
        lwumi,
        LTRX_USER_MESSAGE_SEVERITY__INFORMATIONAL,
        "Connecting, hit any key to cancel ..."
    );
    ltrx_thread_sleep(2);

    struct gatt_connect_context gcc = { 0 };
    if (
        ltrx_gatt_connect(
            bda, transport, type, gatt_connect_callback, &gcc
        ) != 0
    )
    {
        ltrx_write_user_message(
            lwumi,
            LTRX_USER_MESSAGE_SEVERITY__ERROR,
            "Connection failed."
        );
    }
    else
    {
        while (! gcc.complete)
        {
            if (ltrx_input_stream_peek_with_block_time(is, 100) >= 0)
            {
                ltrx_input_stream_read(is);
                if (! ltrx_gatt_connect_cancel(bda))
                {
                    ltrx_write_user_message(
                        lwumi,
                        LTRX_USER_MESSAGE_SEVERITY__ERROR,
                        "Failed to cancel connection."
                    );
                }
                else
                {
                    ltrx_write_user_message(
                        lwumi,
                        LTRX_USER_MESSAGE_SEVERITY__INFORMATIONAL,
                        "Connection cancelled."
                    );
                }
                gcc.complete = CONNECT__ABORT;
                break;
            }
        }
        if (gcc.complete != CONNECT__ABORT)
        {
            if (! gcc.connected)
            {
                ltrx_write_user_message(
                    lwumi,
                    LTRX_USER_MESSAGE_SEVERITY__ERROR,
                    "Connection failed."
                );
            }
            else
            {
                stream_printf(
                    os, "Connection id: %u\r\n\r\n", gcc.connection_id
                );
            }
        }
    }
    return true;
}

bool GATTDisconnect(
    uint32_t zeroBasedIndex,
    const char *value,
    const struct ltrx_write_user_message_info *lwumi
)
{
    (void)zeroBasedIndex;
    VERIFY_CLI_ENVIRONMENT_OR_ABORT(lwumi);

    struct buffer_arg_context bac;
    uint32_t argc;
    if (! buffer_arg_init(value, &bac, &argc))
    {
        return false;
    }
    if (argc < 1)
    {
        ltrx_write_user_message(
            lwumi,
            LTRX_USER_MESSAGE_SEVERITY__ERROR,
            "Must specify connection id."
        );
        buffer_arg_term(&bac);
        return false;
    }
    uint16_t cid = strtoul(buffer_arg_next(&bac), NULL, 0);
    buffer_arg_term(&bac);
    if (! ltrx_gatt_disconnect(cid))
    {
        ltrx_write_user_message(
            lwumi,
            LTRX_USER_MESSAGE_SEVERITY__ERROR,
            "Disconnect failed."
        );
        return false;
    }
    return true;
}

bool GATTSetMTU(
    uint32_t zeroBasedIndex,
    const char *value,
    const struct ltrx_write_user_message_info *lwumi
)
{
    (void)zeroBasedIndex;
    VERIFY_CLI_ENVIRONMENT_OR_ABORT(lwumi);

    struct buffer_arg_context bac;
    uint32_t argc;
    if (! buffer_arg_init(value, &bac, &argc))
    {
        return false;
    }
    if (argc < 2)
    {
        ltrx_write_user_message(
            lwumi,
            LTRX_USER_MESSAGE_SEVERITY__ERROR,
            "Must specify connection id and MTU size."
        );
        buffer_arg_term(&bac);
        return false;
    }
    uint16_t cid = strtoul(buffer_arg_next(&bac), NULL, 0);
    uint16_t mtu = strtoul(buffer_arg_next(&bac), NULL, 0);
    buffer_arg_term(&bac);

    if (! ltrx_gatt_configure_mtu(cid, mtu, TIME_WAIT_FOREVER))
    {
        ltrx_write_user_message(
            lwumi,
            LTRX_USER_MESSAGE_SEVERITY__ERROR,
            "Failed to set MTU."
        );
    }
    return true;
}

bool GATTReadByHandle(
    uint32_t zeroBasedIndex,
    const char *value,
    const struct ltrx_write_user_message_info *lwumi
)
{
    (void)zeroBasedIndex;
    VERIFY_CLI_ENVIRONMENT_OR_ABORT(lwumi);

    struct output_stream *os = ltrx_cli_get_output_stream(
        lwumi->optCliThreadInfo
    );
    struct buffer_arg_context bac;
    uint32_t argc;
    static const char *options = "b:o:";
    char *optarg;
    int c;
    size_t buffer_size = BUFFER_SIZE__DEFAULT;
    uint16_t offset = 0;

    if (! buffer_arg_init(value, &bac, &argc))
    {
        return false;
    }
    if (argc < 2)
    {
        ltrx_write_user_message(
            lwumi,
            LTRX_USER_MESSAGE_SEVERITY__ERROR,
            "Must specify connection id and handle."
        );
        buffer_arg_term(&bac);
        return false;
    }
    while ((c = buffer_arg_getopt(&bac, options, &optarg)) != -1)
    {
        switch (c)
        {
            case 'b':
                buffer_size = strtoul(optarg, NULL, 0);
                if (buffer_size < BUFFER_SIZE__MIN || buffer_size > BUFFER_SIZE__MAX)
                {
                    buffer_size = BUFFER_SIZE__DEFAULT;
                    stream_printf(os, "Defaulting buffer size to %u.\r\n", buffer_size);
                }
                break;

            case 'o':
                offset = (uint16_t)strtoul(optarg, NULL, 0);
                break;

            default:
                break;
        }
    }
    uint16_t cid = (uint16_t)strtoul(buffer_arg_next(&bac), NULL, 0);
    uint16_t handle = (uint16_t)strtoul(buffer_arg_next(&bac), NULL, 0);
    buffer_arg_term(&bac);

    uint16_t length = (uint16_t)buffer_size;
    uint8_t *buffer = MALLOC_ZEROED(buffer_size);
    if (! buffer)
    {
        ltrx_write_user_message(
            lwumi,
            LTRX_USER_MESSAGE_SEVERITY__ERROR,
            "Failed to allocate read buffer."
        );
        return false;
    }

    stream_printf(
        os, 
        "Reading attribute value with buffer size %u.\r\n", 
        buffer_size
    );
    enum ltrx_gatt_status gs = ltrx_gatt_read_by_handle(
        cid, handle, buffer, offset, &length, TIME_WAIT_FOREVER
    );
    bool rc = gs == 0;
    if (! rc)
    {
        stream_printf(
            os, "Read failed: %s.\r\n", str_gatt_status(gs)
        );
    }
    else
    {
        stream_printf(os, "Read %u bytes:\r\n\r\n", length);
        if (length)
        {
            ltrx_output_stream_hexdump(os, buffer, length, 0);
        }
    }
    ltrx_free(buffer);
    return rc;
}

#if 0
bool GATTReadByType(
    uint32_t zeroBasedIndex,
    const char *value,
    const struct ltrx_write_user_message_info *lwumi
)
{
    (void)zeroBasedIndex;
    VERIFY_CLI_ENVIRONMENT_OR_ABORT(lwumi);

    struct output_stream *os = ltrx_cli_get_output_stream(
        lwumi->optCliThreadInfo
    );
    struct buffer_arg_context bac;
    uint32_t argc;
    static const char *options = "b:e:s:";
    char *optarg;
    char *p;
    int c;
    size_t buffer_size = BUFFER_SIZE__DEFAULT;
    struct ltrx_gatt_handle_range hr = {
        .h_start = 0x1,
        .h_end = 0xFFFF,
    };

    if (! buffer_arg_init(value, &bac, &argc))
    {
        return false;
    }
    if (argc < 2)
    {
        ltrx_write_user_message(
            lwumi,
            LTRX_USER_MESSAGE_SEVERITY__ERROR,
            "Must specify connection id, handle range, and UUID."
        );
        buffer_arg_term(&bac);
        return false;
    }
    while ((c = buffer_arg_getopt(&bac, options, &optarg)) != -1)
    {
        switch (c)
        {
            case 'b':
                buffer_size = strtoul(optarg, NULL, 0);
                if (buffer_size < BUFFER_SIZE__MIN || buffer_size > BUFFER_SIZE__MAX)
                {
                    buffer_size = BUFFER_SIZE__DEFAULT;
                }
                break;

            case 'e':
                hr.h_end = (uint16_t)strtoul(optarg, NULL, 0);
                break;

            case 's':
                hr.h_start = (uint16_t)strtoul(optarg, NULL, 0);
                break;

            default:
                break;
        }
    }

    struct ltrx_uuid lu;
    uint16_t cid = (uint16_t)strtoul(buffer_arg_next(&bac), NULL, 0);
    bt_uuid_parse(buffer_arg_next(&bac), &lu);
    buffer_arg_term(&bac);

    uint16_t length = (uint16_t)buffer_size;
    uint8_t *buffer = MALLOC_ZEROED(buffer_size);
    if (! buffer)
    {
        ltrx_write_user_message(
            lwumi,
            LTRX_USER_MESSAGE_SEVERITY__ERROR,
            "Failed to allocate read buffer."
        );
        return false;
    }
    enum ltrx_gatt_status gs = ltrx_gatt_read_by_type(
        cid, &hr, &lu, buffer, &length, TIME_WAIT_FOREVER
    );
    bool rc = gs == 0;
    if (! rc)
    {
        stream_printf(
            os, "Read failed: %s.\r\n", str_gatt_status(gs)
        );
    }
    else
    {
        stream_printf(os, "Read %u bytes:\r\n\r\n", length);
        if (length)
        {
            ltrx_output_stream_hexdump(os, buffer, length, 0);
        }
    }
    ltrx_free(buffer);
    return rc;
}
#endif

static const char *str_write_type(uint8_t wt)
{
    switch (wt)
    {
    default:
    case WRITE_TYPE__NORMAL: return "Write";
    case WRITE_TYPE__NO_RESPONSE: return "Write with no response";
    case WRITE_TYPE__PREPARE: return "Write prepare";
    case WRITE_TYPE__CANCEL: return "Write cancel";
    case WRITE_TYPE__EXECUTE: return "Write execute";
    }
}

bool GATTWrite(
    uint32_t zeroBasedIndex,
    const char *value,
    const struct ltrx_write_user_message_info *lwumi
)
{
    (void)zeroBasedIndex;
    VERIFY_CLI_ENVIRONMENT_OR_ABORT(lwumi);
    struct output_stream *os = ltrx_cli_get_output_stream(
        lwumi->optCliThreadInfo
    );
    struct buffer_arg_context bac;
    uint32_t argc;
    static const char *options = "ceo:pr";
    char *optarg;
    int c;
    uint16_t handle = 0;
    uint16_t offset = 0;
    uint16_t length = 0;
    uint8_t *data = NULL;
    uint8_t write_type = WRITE_TYPE__NORMAL;

    if (! buffer_arg_init(value, &bac, &argc))
    {
        return false;
    }

    while ((c = buffer_arg_getopt(&bac, options, &optarg)) != -1)
    {
        switch (c)
        {
            case 'c':
                write_type = WRITE_TYPE__CANCEL;
                break;
            case 'e':
                write_type = WRITE_TYPE__EXECUTE;
                break;
            case 'o':
                offset = (uint16_t)strtoul(optarg, NULL, 0);
                break;
            case 'p':
                write_type = WRITE_TYPE__PREPARE;
                break;
            case 'r':
                write_type = WRITE_TYPE__NO_RESPONSE;
                break;

            default:
                break;
        }
    }
    if (
        (
            write_type == WRITE_TYPE__NORMAL ||
            write_type == WRITE_TYPE__NO_RESPONSE ||
            write_type == WRITE_TYPE__PREPARE
        ) &&
        argc < 3
    )
    {
        ltrx_write_user_message(
            lwumi,
            LTRX_USER_MESSAGE_SEVERITY__ERROR,
            "Must specify connection id, handle, and data."
        );
        buffer_arg_term(&bac);
        return false;
    }
    else if (argc < 1)
    {
        ltrx_write_user_message(
            lwumi,
            LTRX_USER_MESSAGE_SEVERITY__ERROR,
            "Must specify connection id."
        );
        buffer_arg_term(&bac);
        return false;
    }

    optarg = buffer_arg_next(&bac);
    if (! *optarg)
    {
        ltrx_write_user_message(
            lwumi,
            LTRX_USER_MESSAGE_SEVERITY__ERROR,
            "Must specify connection id."
        );
        buffer_arg_term(&bac);
        return false;
    }
    if (offset != 0 && write_type != WRITE_TYPE__PREPARE)
    {
        ltrx_write_user_message(
            lwumi,
            LTRX_USER_MESSAGE_SEVERITY__INFORMATIONAL,
            "Ignoring offset for non-prepare write."
        );
    }

    uint16_t cid = (uint16_t)strtoul(optarg, NULL, 0);
    enum ltrx_gatt_status gs;
    if (
        write_type == WRITE_TYPE__NORMAL ||
        write_type == WRITE_TYPE__NO_RESPONSE ||
        write_type == WRITE_TYPE__PREPARE
    )
    {
        handle = (uint16_t)strtoul(buffer_arg_next(&bac), NULL, 0);
        uint32_t n;
        char *p = buffer_arg_next(&bac);
        if (p[0] == '-' && p[1] == 'n')
        {
            if (p[2] && p[2] != ' ')
            {
                length = strtoul(&p[2], NULL, 0);
            }
            else
            {
                length = 4;
            }
            char *s = buffer_arg_next(&bac);
            n = strtoul(s, NULL, 0);
            data = (uint8_t *)&n;
        }
        else
        {
            length = strlen(p);
            data = (uint8_t *)p;
        }

        gs = ltrx_gatt_write(cid, handle, data, offset, length, write_type);
    }
    else if (write_type == WRITE_TYPE__EXECUTE)
    {
        gs = ltrx_gatt_prepared_write_execute(cid);
    }
    else
    {
        gs = ltrx_gatt_prepared_write_cancel(cid);
    }
    if (gs != 0)
    {
        stream_printf(
            os,
            "%s failed: %s\r\n",
            str_write_type(write_type),
            str_gatt_status(gs)
        );
    }
    buffer_arg_term(&bac);
    return gs == 0;
}

static void notification_callback(
    uint16_t connection_id,
    uint16_t handle,
    uint8_t *data,
    uint16_t offset,
    uint16_t length,
    void *opaque
)
{
#pragma GCC diagnostic push
#pragma GCC diagnostic ignored"-Wcast-align"
    struct bts_status *status = (struct bts_status *)data;
#pragma GCC diagnostic pop
    if (s_notification_listen && s_notification_ostream)
    {
        struct output_stream *os = s_notification_ostream;
        stream_printf(
            os, 
            "Notification connection id %u handle  0x%X", 
            connection_id, handle
        );
        struct ltrx_time lt;
        ltrx_time_to_local(status->time, &lt);
        stream_printf(
            os,
            "Time: %04d-%02d-%02d %02d:%02d:%02d",
            lt.t_year, lt.t_month, lt.t_dayofmonth,
            lt.t_hour, lt.t_minute, lt.t_second
        );

        struct bts_stats *bss = &status->stats;
        stream_printf(os, "  Connection ID:   %u\r\n", bss->bs_connection_id);
        stream_printf(os, "  MTU:             %u\r\n", bss->bs_mtu);
        stream_printf(os, "  Bytes written:   %u\r\n", bss->bs_nwritten);
        stream_printf(os, "  Queued writes:   %u\r\n", bss->bs_queued_writes);
        stream_printf(os, "  Queued bytes:    %u\r\n", bss->bs_queued_write_bytes);
        stream_printf(os, "  Bytes read:      %u\r\n", bss->bs_nread);
        stream_printf(os, "\r\n");
        ltrx_output_stream_flush_data(os);
    }
    else
    {
        struct ltrx_time lt;
        ltrx_time_to_local(status->time, &lt);
        TLOG(
            TLOG_SEVERITY_LEVEL__INFORMATIONAL,
            "Time: %04d-%02d-%02d %02d:%02d:%02d\r\n",
            lt.t_year, lt.t_month, lt.t_dayofmonth,
            lt.t_hour, lt.t_minute, lt.t_second
        );
        struct bts_stats *bss = &status->stats;
        TLOG(
            TLOG_SEVERITY_LEVEL__INFORMATIONAL,
            "GATT notification conn id %u handle 0x%X:",
            connection_id, handle
        );
        TLOG(
            TLOG_SEVERITY_LEVEL__INFORMATIONAL,
            "  Connection ID:   %u\r\n", bss->bs_connection_id
        );
        TLOG(
            TLOG_SEVERITY_LEVEL__INFORMATIONAL,
            "  MTU:             %u\r\n", bss->bs_mtu
        );
        TLOG(
            TLOG_SEVERITY_LEVEL__INFORMATIONAL,
            "  Bytes written:   %u\r\n", bss->bs_nwritten
        );
        TLOG(
            TLOG_SEVERITY_LEVEL__INFORMATIONAL,
            "  Queued writes:   %u\r\n", bss->bs_queued_writes
        );
        TLOG(
            TLOG_SEVERITY_LEVEL__INFORMATIONAL,
            "  Queued bytes:    %u\r\n", bss->bs_queued_write_bytes
        );
        TLOG(
            TLOG_SEVERITY_LEVEL__INFORMATIONAL,
            "  Bytes read:      %u\r\n", bss->bs_nread
        );
    }
}

static void indication_callback(
    uint16_t connection_id,
    uint16_t handle,
    uint8_t *data,
    uint16_t offset,
    uint16_t length,
    void *opaque
)
{
#pragma GCC diagnostic push
#pragma GCC diagnostic ignored"-Wcast-align"
    struct bts_status *status = (struct bts_status *)data;
#pragma GCC diagnostic pop
    if (s_notification_listen && s_notification_ostream)
    {
        struct output_stream *os = s_notification_ostream;
        stream_printf(
            os,
            "Indication connection id %u handle 0x%X\r\n",
            connection_id, handle
        );
        struct ltrx_time lt;
        ltrx_time_to_local(status->time, &lt);
        stream_printf(
            os,
            "Time: %04d-%02d-%02d %02d:%02d:%02d",
            lt.t_year, lt.t_month, lt.t_dayofmonth,
            lt.t_hour, lt.t_minute, lt.t_second
        );

        struct bts_stats *bss = &status->stats;
        stream_printf(os, "  Connection ID:   %u\r\n", bss->bs_connection_id);
        stream_printf(os, "  MTU:             %u\r\n", bss->bs_mtu);
        stream_printf(os, "  Bytes written:   %u\r\n", bss->bs_nwritten);
        stream_printf(os, "  Queued writes:   %u\r\n", bss->bs_queued_writes);
        stream_printf(os, "  Queued bytes:    %u\r\n", bss->bs_queued_write_bytes);
        stream_printf(os, "  Bytes read:      %u\r\n", bss->bs_nread);
        stream_printf(os, "\r\n");
        ltrx_output_stream_flush_data(os);
    }
    else
    {
        TLOG(
            TLOG_SEVERITY_LEVEL__INFORMATIONAL,
            "GATT indication conn id %u handle 0x%X:\r\n",
            connection_id, handle
        );
        struct ltrx_time lt;
        ltrx_time_to_local(status->time, &lt);
        TLOG(
            TLOG_SEVERITY_LEVEL__INFORMATIONAL,
            "Time: %04d-%02d-%02d %02d:%02d:%02d\r\n",
            lt.t_year, lt.t_month, lt.t_dayofmonth,
            lt.t_hour, lt.t_minute, lt.t_second
        );
        struct bts_stats *bss = &status->stats;
        TLOG(
            TLOG_SEVERITY_LEVEL__INFORMATIONAL,
            "  Connection ID:   %u\r\n", bss->bs_connection_id
        );
        TLOG(
            TLOG_SEVERITY_LEVEL__INFORMATIONAL,
            "  MTU:             %u\r\n", bss->bs_mtu
        );
        TLOG(
            TLOG_SEVERITY_LEVEL__INFORMATIONAL,
            "  Bytes written:   %u\r\n", bss->bs_nwritten
        );
        TLOG(
            TLOG_SEVERITY_LEVEL__INFORMATIONAL,
            "  Queued writes:   %u\r\n", bss->bs_queued_writes
        );
        TLOG(
            TLOG_SEVERITY_LEVEL__INFORMATIONAL,
            "  Queued bytes:    %u\r\n", bss->bs_queued_write_bytes
        );
        TLOG(
            TLOG_SEVERITY_LEVEL__INFORMATIONAL,
            "  Bytes read:      %u\r\n", bss->bs_nread
        );
    }
    ltrx_gatt_confirm_indication(connection_id, handle);
}

static bool notification_indication_action(
    bool notification,
    const char *value,
    const struct ltrx_write_user_message_info *lwumi
)
{
    VERIFY_CLI_ENVIRONMENT_OR_ABORT(lwumi);
    struct output_stream *os = ltrx_cli_get_output_stream(
        lwumi->optCliThreadInfo
    );
    struct input_stream *is = ltrx_cli_get_input_stream(
        lwumi->optCliThreadInfo
    );
    struct buffer_arg_context bac;
    uint32_t argc;
    static const char *options = "b:e:s:";
    char *optarg;
    int c;
    struct ltrx_gatt_handle_range hr = {
        .h_start = 0x1,
        .h_end = 0xFFFF,
    };

    if (! buffer_arg_init(value, &bac, &argc))
    {
        return false;
    }
    while ((c = buffer_arg_getopt(&bac, options, &optarg)) != -1)
    {
        switch (c)
        {
            case 'e':
                hr.h_end = (uint16_t)strtoul(optarg, NULL, 0);
                break;

            case 's':
                hr.h_start = (uint16_t)strtoul(optarg, NULL, 0);
                break;

            default:
                break;
        }
    }

    optarg = buffer_arg_next(&bac);
    if (strcmp(optarg, "on") == 0)
    {
        if (notification)
        {
            ltrx_gatt_notification_register(
                &hr, notification_callback, NULL
            );
        }
        else
        {
            ltrx_gatt_indication_register(
                &hr, indication_callback, NULL
            );
        }
    }
    else if (strcmp(optarg, "off") == 0)
    {
        if (notification)
        {
            ltrx_gatt_notification_unregister(
                notification_callback
            );
        }
        else
        {
            ltrx_gatt_indication_unregister(
                indication_callback
            );
        }
    }
    else if (strcmp(optarg, "listen") == 0)
    {
        ltrx_write_user_message(
            lwumi,
            LTRX_USER_MESSAGE_SEVERITY__INFORMATIONAL,
            "Listening for notifications, hit any key to cancel.\r\n"
        );
        s_notification_listen = true;
        s_notification_ostream = os;
        do {
            if (ltrx_input_stream_peek_with_block_time(is, 100) >= 0)
            {
                ltrx_input_stream_read(is);
                break;
            }
        } while (true);
        s_notification_ostream = NULL;
        s_notification_listen = false;
    }
    return true;
}

bool GATTNotification(
    uint32_t zeroBasedIndex,
    const char *value,
    const struct ltrx_write_user_message_info *lwumi
)
{
    (void)zeroBasedIndex;
    return notification_indication_action(
        true,
        value,
        lwumi
    );
}

bool GATTIndication(
    uint32_t zeroBasedIndex,
    const char *value,
    const struct ltrx_write_user_message_info *lwumi
)
{
    (void)zeroBasedIndex;
    return notification_indication_action(
        false,
        value,
        lwumi
    );
}
